/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_delegate_nsfile.mm
 *  \brief      MGL ファイルデリゲート Apple環境向け
 *  \date       Since: August 7, 2023. 15:48:32 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/file/delegate/mgl_file_delegate_nsfile.h>
#if defined(MGL_FILE_DELEGATE_ENABLE_NSFILE)

#include <cerrno>

#import <Foundation/NSFileManager.h>
#import <Foundation/NSString.h>

namespace MGL::File
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルをオープン
 *  \param[out]     result      処理結果
 *  \param[in,out]  mountWork   マウント毎のワーク
 *  \param[in]      path        オープンするファイルのパス
 *  \param[in]      mode        オープンモード
 *  \return         オープンしたファイルのワーク
 */
/* ------------------------------------------------------------------------- */
AccessWorkPtr NSFileDelegate::Open(Result &result, [[maybe_unused]] SharedMountWork &mountWork, const PathView &path, OpenMode mode) noexcept
{
    // 引数チェック
    if ((mode != OpenMode::Read) && (mode != OpenMode::Write))
    {
        result = Error::InvalidArgument;
        return nullptr;
    }
    
    NSFileManager *manager = NSFileManager.defaultManager;
    NSString *nsPath = [NSString stringWithUTF8String: path.GetCString()];

    // 読み込み時のチェック
    if (mode == OpenMode::Read)
    {
        // ファイルが存在しない場合はエラー
        if (![manager fileExistsAtPath: nsPath])
        {
            result = Error::FileNotExist;
            return nullptr;
        }
    }

    // ファイルの情報を取得
    NSError *error = nil;
    NSDictionary *attributes = [manager attributesOfItemAtPath: nsPath
                                                         error: &error];
    if (attributes == nil)
    {
        result = GetError(error);
        return nullptr;
    }

    // 通常ファイルでなければ失敗
    if (attributes.fileType != NSFileTypeRegular)
    {
        result = Error::NoRegularFile;
        return nullptr;
    }
    
    // ファイルをオープン
    FILE *fp = fopen(path, GetModeString(mode));
    if (fp == nullptr)
    {
        result = GetError(errno);
        return nullptr;
    }

    // ワークに値を設定
    auto work = STL::make_unique<NSFileWork>(mode);
    work->fp = fp;
    work->size = size_t(attributes.fileSize);
    
    result.Success();

    return work;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルをクローズ
 *  \param[in,out]  work    クローズするファイルのワーク
 *  \return         エラー発生時にいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result NSFileDelegate::Close(AccessWork *work) noexcept
{
    auto *nsWork = GetWork(work);
    if (nsWork == nullptr)
    {
        return Error::InvalidArgument;
    }
    
    fclose(nsWork->fp);
    nsWork->fp = nullptr;
    
    return Result::Succeeded();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルを読み込み
 *  \param[in,out]  work        アクセスするファイルのワーク
 *  \param[out]     result      処理結果
 *  \param[in]      buffer      読み込み先のアドレス
 *  \param[in]      size        読み込みサイズ
 *  \return         実際に読み込んだバイト数
 */
/* ------------------------------------------------------------------------- */
size_t NSFileDelegate::Read(AccessWork *work, Result &result, void *buffer, size_t size) noexcept
{
    auto *nsWork = GetWork(work, OpenMode::Read);
    if (nsWork == nullptr)
    {
        result = Error::InvalidArgument;
        return 0;
    }

    clearerr(nsWork->fp);
    auto readSize = fread(buffer, 1, size, nsWork->fp);

    if (ferror(nsWork->fp) != 0)
    {
        result = Error::ReadError;
    }

    result.Success();
    return readSize;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルを書き込み
 *  \param[in,out]  work        アクセスするファイルのワーク
 *  \param[out]     result      処理結果
 *  \param[out]     buffer      書き込み先のアドレス
 *  \param[in]      size        書き込みサイズ
 *  \return         実際に書き込んだバイト数
 */
/* ------------------------------------------------------------------------- */
size_t NSFileDelegate::Write(AccessWork *work, Result &result, const void *buffer, size_t size) noexcept
{
    auto *nsWork = GetWork(work, OpenMode::Write);
    if (nsWork == nullptr)
    {
        result = Error::InvalidArgument;
        return 0;
    }

    clearerr(nsWork->fp);
    auto writeSize = fwrite(buffer, 1, size, nsWork->fp);
    if (writeSize < size)
    {
        result = Error::WriteError;
    }
    else
    {
        result.Success();
    }
    
    return writeSize;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ストリーム位置を設定
 *  \param[in,out]  work        アクセスするファイルのワーク
 *  \param[out]     result      エラー発生時の結果の格納先
 *  \param[in]      seekType    シークタイプ
 *  \param[in]      offset      オフセット
 *  \return         エラー発生時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
size_t NSFileDelegate::Seek(AccessWork *work, Result &result, SeekType seekType, int32_t offset) noexcept
{
    auto *nsWork = GetWork(work);
    if (nsWork == nullptr)
    {
        result = Error::InvalidArgument;
        return 0;
    }

    int seek = SEEK_SET;    
    switch (seekType)
    {
        case SeekType::Top:
            seek = SEEK_SET;
            break;
            
        case SeekType::End:
            seek = SEEK_END;
            break;
            
        case SeekType::Current:
            seek = SEEK_CUR;
            break;
    }
    
    if (fseek(nsWork->fp, offset, seek) != 0)
    {
        result = GetError(errno);
    }
    else
    {
        result.Success();
    }

    return static_cast<size_t>(ftell(nsWork->fp));
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ストリーム位置を取得
 *  \param[in]  work    アクセスするファイルのワーク
 *  \param[out] result  エラー発生時の結果の格納先
 *  \return     現在のストリーム位置
 */
/* ------------------------------------------------------------------------- */
size_t NSFileDelegate::GetOffset(AccessWork *work, Result &result) const noexcept
{
    auto *nsWork = GetWork(work);
    if (nsWork == nullptr)
    {
        result = Error::InvalidArgument;
        return 0;
    }
    
    auto offset = ftell(nsWork->fp);
    if (offset < 0)
    {
        result = Error::UnknownError;
        return 0;
    }

    result.Success();
    return static_cast<size_t>(offset);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルストリームが終端に達しているかを取得
 *  \param[in,out]  work    アクセスするファイルのワーク
 *  \param[out]     result  エラー発生時の結果の格納先
 *  \retval         true    終端に達している
 *  \retval         false   終端に達していない
 */
/* ------------------------------------------------------------------------- */
bool NSFileDelegate::IsEOF(AccessWork *work, Result &result) const noexcept
{
    auto *nsWork = GetWork(work);
    if (nsWork == nullptr)
    {
        result = Error::InvalidArgument;
        return false;
    }

    auto offset = GetOffset(work, result);
    if (result.HasError())
    {
        return false;
    }
    if (offset >= nsWork->size)
    {
        return true;
    }
    
    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      オープンしているファイルのサイズを取得
 *  \param[in]  work    アクセスするファイルのワーク
 *  \param[out] result  エラー発生時の結果の格納先
 *  \return     ファイルサイズ
 */
/* ------------------------------------------------------------------------- */
size_t NSFileDelegate::GetSize(AccessWork *work, Result &result) const noexcept
{
    auto *nsWork = GetWork(work);
    if (nsWork == nullptr)
    {
        result = Error::InvalidArgument;
        return 0;
    }

    auto offset = ftell(nsWork->fp);
    fseek(nsWork->fp, 0, SEEK_END);
    auto size = ftell(nsWork->fp);
    fseek(nsWork->fp, offset, SEEK_SET);
    
    if (size < 0)
    {
        result = Error::UnknownError;
        return 0;
    }

    result.Success();
    return static_cast<size_t>(size);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルのサイズを取得
 *  \param[in,out]  mountWork   マウント毎のワーク
 *  \param[out]     result      エラー発生時の結果の格納先
 *  \param[in]      path        サイズを取得するファイルのパス
 *  \return         ファイルサイズ
 */
/* ------------------------------------------------------------------------- */
size_t NSFileDelegate::GetSize([[maybe_unused]] MountWork *mountWork, Result &result, const PathView &path) const noexcept
{
    NSFileManager *manager = NSFileManager.defaultManager;
    NSString *nsPath = [NSString stringWithUTF8String: path.GetCString()];

    // ファイルの情報を取得
    NSError *error = nil;
    NSDictionary *attributes = [manager attributesOfItemAtPath: nsPath
                                                         error: &error];
    if (attributes == nil)
    {
        result = GetError(error);
        return 0;
    }

    result.Success();
    return size_t(attributes.fileSize);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ディレクトリを作成する
 *  \param[in,out]  mountWork   マウント毎のワーク
 *  \param[in]      path        作成するディレクトリのパス
 *  \return         エラー発生時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result NSFileDelegate::MakeDirectory([[maybe_unused]] MountWork *mountWork, const PathView &path) noexcept
{
    NSString *dirPath = [NSString stringWithUTF8String:path];
    
    auto *manager = NSFileManager.defaultManager;

    // 既に存在しているかをチェック
    BOOL isDirectory = NO;
    auto existsFile = [manager fileExistsAtPath:dirPath
                               isDirectory:&isDirectory];
    if (existsFile)
    {
        // ディレクトリが既に存在しているのであれば成功
        if (isDirectory)
        {
            return Result::Succeeded();
        }
        // ディレクトリでないファイルが存在していたら失敗
        else
        {
            return Error::FileAlreadyExist;
        }
    }
    
    // ディレクトリを生成
    NSError *error = nil;
    auto isSucceeded = [manager createDirectoryAtPath:dirPath
                          withIntermediateDirectories:NO
                                           attributes:nil
                                                error:&error];
    // 失敗していたらエラーコードを取得して返す
    if (!isSucceeded)
    {
        return GetError(error);
    }
    
    return Result::Succeeded();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルの移動・リネーム
 *  \param[in,out]  mountWork   マウント毎のワーク
 *  \param[in]      sourcePath  移動元のパス
 *  \param[in]      destPath    移動先のパス
 *  \return         エラー発生時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result NSFileDelegate::Move([[maybe_unused]] MountWork *mountWork, const PathView &sourcePath, const PathView &destPath) noexcept
{
    int result = rename(sourcePath, destPath);
    if (result == -1)
    {
        return GetError(errno);
    }

    return Result::Succeeded();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルの削除
 *  \param[in,out]  mountWork   マウント毎のワーク
 *  \param[in]      path        削除するファイルのパス
 *  \return         エラー発生時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result NSFileDelegate::Remove([[maybe_unused]] MountWork *mountWork, const PathView &path) noexcept
{
    int result = remove(path);
    if (result != 0)
    {
        return GetError(errno);
    }

    return Result::Succeeded();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルの存在をチェック
 *  \param[in,out]  mountWork   マウント毎のワーク
 *  \param[out]     result      エラー発生時の結果の格納先
 *  \param[in]      path        チェックするファイルのパス
 *  \return         存在する場合はエラーなしのリザルトが返る
 */
/* ------------------------------------------------------------------------- */
bool NSFileDelegate::Exists([[maybe_unused]] MountWork *mountWork, Result &result, const PathView &path) noexcept
{
    NSFileManager *manager = NSFileManager.defaultManager;
    NSString *nsPath = [NSString stringWithUTF8String: path.GetCString()];

    result.Success();
    
    return [manager fileExistsAtPath: nsPath];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      作業用構造体からこのクラス用の構造体を取得する
 *  \param[in]  work    作業用構造体
 *  \param[in]  mode    オープンモード指定．OpenMode::Noneを指定した場合は無視される
 *  \return     作業用構造体がこのクラス用に作られていれば有効なアドレスを返す
 */
/* ------------------------------------------------------------------------- */
NSFileDelegate::NSFileWork *NSFileDelegate::GetWork(AccessWork *work, OpenMode mode) const noexcept
{
    if (work != nullptr)
    {
        if (work->IsOpen())
        {
            if ((mode == OpenMode::None) || (mode == work->GetOpenMode()))
            {
                if (work->GetDelegateKey() == kDelegateKey)
                {
                    return static_cast<NSFileWork *>(work);
                }
            }
        }
    }
    
    return nullptr;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      オープンモードからモード指定用の文字列を取得
 *  \param[in]  mode    オープンモード
 *  \return     オープンモードに対応した文字列
 */
/* ------------------------------------------------------------------------- */
const char *NSFileDelegate::GetModeString(OpenMode mode) const noexcept
{
    static const char *modeReadBinary = "rb";
    static const char *modeWriteBinary = "wb";

    switch (mode)
    {
        case OpenMode::Read:
            return modeReadBinary;

        case OpenMode::Write:
            return modeWriteBinary;

        default:
            break;
    }

    return nullptr;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      エラー番号から処理結果を取得する
 *  \param[in]  errorNumber     ファイル操作後のerrno
 *  \return     エラー番号に対応した処理結果
 */
/* ------------------------------------------------------------------------- */
Error NSFileDelegate::GetError(int errorNumber) const noexcept
{
    switch (errorNumber)
    {
        case ENOENT:
            return Error::FileNotExist;
            
        case EPERM:
        case EACCES:
            return Error::NotPermitted;
            
        case EINVAL:
            return Error::InvalidArgument;
            
        default:
            break;
    }
    
    return Error::UnknownError;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      NSErrorから処理結果を取得する
 *  \param[in]  nsError     NSErrorクラス
 *  \return     NSErrorが保持しているエラー番号に対応したエラー
 *  \note
 *      POSIXエラードメイン以外はError::UnknownError扱い．
 *      ファイル関連の処理で他のエラードメインが使用されるかは要調査．
 */
/* ------------------------------------------------------------------------- */
Error NSFileDelegate::GetError(const NSError *nsError) const noexcept
{
    if (nsError == nil)
    {
        return Error::None;
    }
    
    if (nsError.domain == NSPOSIXErrorDomain)
    {
        return GetError(int(nsError.code));
    }
    
    return Error::UnknownError;
}
}   // namespace MGL::File

#endif  // MGL_FILE_DELEGATE_ENABLE_NSFILE

// vim: et ts=4 sw=4 sts=4
