// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_event_register_list.cc
 *  \brief      MGL イベント登録リスト
 *  \date       Since: December 23, 2020. 13:51:08 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/event/mgl_event_register_list.h>

#include <algorithm>

namespace MGL::Event
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
RegisterList::RegisterList() noexcept
    : _currentThreadId(std::thread::id())
{
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      リストに登録
 *  \param[in]  item    登録アイテム
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool RegisterList::Register(STL::unique_ptr<RegisterItem> item) noexcept
{
    // 実行中は一時登録リストに追加する
    if (IsRunning())
    {
        const std::scoped_lock lock(_temporaryListMutex);
        _temporaryRegisterList.push_back(std::move(item));
    }
    // 実行中でなければ直接追加
    else
    {
        const std::scoped_lock lock(_activeListMutex);
        _activeList.push_back(std::move(item));
    }

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief          リストから登録解除
 *  \param[in,out]  handle  登録を解除するハンドル
 */
/* ------------------------------------------------------------------------- */
void RegisterList::Unregister(Handle &handle) noexcept
{
    // 実行中は一時登録解除リストに追加する
    if (IsRunning())
    {
        // 実行中のイベントを異なるスレッドから削除しようとした場合は待ってもらう
        if (IsRunning(handle.GetUniqueID()) && !IsSameThread())
        {
            std::unique_lock lock(_syncExecuteMutex);
            _syncExecute.wait(lock, [&]
            {
                return !IsRunning(handle.GetUniqueID());
            });
        }

        const std::scoped_lock lock(_temporaryListMutex);
        _temporaryUnregisterList.push_back(std::move(handle));
    }
    // 実行中でなければ直接削除
    else
    {
        const std::scoped_lock lock(_activeListMutex);

        for (auto it = _activeList.begin(); it != _activeList.end(); ++it)
        {
            if (handle.GetUniqueID() == (*it)->id)
            {
                _activeList.erase(it);
                handle.Set(NotifyType::Reserve_Invalid, 0);
                break;
            }
        }
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      登録済みのイベントを実行
 *  \param[in]  notifyArg   通知の引数（コールバック関数のnotifyArgに送られる）
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool RegisterList::Execute(void *notifyArg) noexcept
{
    // イベントの多重通知は行わない
    if (IsRunning())
    {
        return false;
    }

    // 実行
    _currentThreadId = std::this_thread::get_id();
    _activeListMutex.lock();
    for (const auto &it : _activeList)
    {
        // 実行しようとしているイベントが登録解除されている場合は実行しない
        if (IsRequestedUnregister(it->id))
        {
            continue;
        }

        // 実行するイベントのIDを保持
        {
            const std::unique_lock lock(_syncExecuteMutex);
            _executeUniqueId = it->id;
        }

        // コールバック関数を呼び出す
        it->callback(it->callbackArg, notifyArg);

        // 実行しているIDをクリアして起床を通知する
        {
            const std::unique_lock lock(_syncExecuteMutex);
            _executeUniqueId = 0;
            _syncExecute.notify_all();
        }
    }
    _activeListMutex.unlock();
    _currentThreadId = std::thread::id();

    // 一時リストを適用
    ApplyTemporaryUnregisterList();
    ApplyTemporaryRegisterList();

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      現在実行中かどうかを返す
 *  \retval     true    実行中
 *  \retval     false   実行中でない
 */
/* ------------------------------------------------------------------------- */
bool RegisterList::IsRunning() const noexcept
{
    return _currentThreadId != std::thread::id();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      指定したIDのイベントが実行中かを返す
 *  \retval     true    実行中
 *  \retval     false   実行中でない
 */
/* ------------------------------------------------------------------------- */
bool RegisterList::IsRunning(UniqueID id) const noexcept
{
    return _executeUniqueId == id;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      実行中のスレッドが現在のスレッドと同じかどうかを返す
 *  \retval     true    同じスレッドで実行中
 *  \retval     false   実行中ではない or 異なるスレッドで実行中
 */
/* ------------------------------------------------------------------------- */
bool RegisterList::IsSameThread() const noexcept
{
    return _currentThreadId == std::this_thread::get_id();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      一時登録リストを適用
 */
/* ------------------------------------------------------------------------- */
void RegisterList::ApplyTemporaryRegisterList() noexcept
{
    const std::scoped_lock temporaryListLock(_temporaryListMutex);
    const std::scoped_lock activeListLock(_activeListMutex);

    for (auto &item : _temporaryRegisterList)
    {
        _activeList.push_back(std::move(item));
    }
    _temporaryRegisterList.clear();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      一時登録解除リストを適用
 */
/* ------------------------------------------------------------------------- */
void RegisterList::ApplyTemporaryUnregisterList() noexcept
{
    const std::scoped_lock temporaryListLock(_temporaryListMutex);
    const std::scoped_lock _activeListLock(_activeListMutex);

    for (auto &unregisterHandle : _temporaryUnregisterList)
    {
        for (auto it = _activeList.begin(); it != _activeList.end(); ++it)
        {
            if (unregisterHandle.GetUniqueID() == (*it)->id)
            {
                _activeList.erase(it);
                unregisterHandle.Set(NotifyType::Reserve_Invalid, 0);
                break;
            }
        }
    }

    _temporaryUnregisterList.clear();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      指定したIDが登録解除を要求されているかを取得
 *  \param[in]  id      イベントID
 *  \retval     true    要求されている
 *  \retval     false   要求されていない
 */
/* ------------------------------------------------------------------------- */
bool RegisterList::IsRequestedUnregister(UniqueID id) noexcept
{
    const std::scoped_lock lock(_temporaryListMutex);

    return std::any_of(_temporaryUnregisterList.begin(),
                       _temporaryUnregisterList.end(),
                       [id](const Handle &handle)
                       {
                           return handle.GetUniqueID() == id;
                       });
}
}    // namespace MGL::Event

// vim: et ts=4 sw=4 sts=4
