// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_event_notifier.cc
 *  \brief      MGL イベント通知
 *  \date       Since: December 23, 2020. 13:34:17 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/event/mgl_event_notifier.h>

namespace MGL::Event
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インスタンスの取得
 *  \return     インスタンスの参照
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Notifier> &Notifier::GetInstanceRef() noexcept
{
    static STL::unique_ptr<Notifier> sInstance = nullptr;
    return sInstance;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
Notifier::Notifier() noexcept
    : _registerListArray()
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          イベント通知の登録
 *  \param[in,out]  handle      登録したハンドルの格納先
 *  \param[in]      type        通知タイプ
 *  \param[in]      callback    コールバック関数
 *  \param[in]      callbackArg コールバック関数の引数
 *  \retval         true        成功
 *  \retval         false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Notifier::Register(Handle &handle, NotifyType type, CallbackFunction callback, void *callbackArg) noexcept
{
    // 既に有効になっているハンドルには登録しない
    if (handle.IsValid())
    {
        return false;
    }

    // 登録リストを取得
    auto *list = GetRegisterList(type);
    if (list == nullptr)
    {
        return false;
    }

    // 登録アイテムのユニークポインタを生成
    auto item = STL::make_unique<RegisterItem>();

    // 登録アイテムにパラメータを設定
    auto id = reinterpret_cast<UniqueID>(item.get());
    item->id = id;
    item->type = type;
    item->callback = callback;
    item->callbackArg = callbackArg;

    // アイテムを登録
    if (!list->Register(std::move(item)))
    {
        return false;
    }

    // ハンドルにパラメータを設定
    handle.Set(type, id);

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          イベント通知の登録解除
 *  \param[in,out]  handle  登録を解除するハンドル
 */
/* ------------------------------------------------------------------------- */
void Notifier::Unregister(Handle &handle) noexcept
{
    // 無効なハンドルは操作しない
    if (!handle.IsValid())
    {
        return;
    }

    // 登録リストの取得
    auto *list = GetRegisterList(handle.GetType());
    if (list == nullptr)
    {
        return;
    }

    // リストから登録解除
    list->Unregister(handle);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      イベント通知の実行
 *  \param[in]  type        通知タイプ
 *  \param[in]  notifyArg   通知の引数（コールバック関数のnotifyArgに送られる）
 *  \retval     true        成功
 *  \retval     false       失敗
 *  \note
 *      同じ通知タイプのイベントを多重に呼ぶことはできず，その場合は失敗となる．
 */
/* ------------------------------------------------------------------------- */
bool Notifier::Notify(NotifyType type, void *notifyArg) noexcept
{
    // 登録リストを取得
    auto *list = GetRegisterList(type);
    if (list == nullptr)
    {
        return false;
    }

    // 実行
    return list->Execute(notifyArg);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      通知タイプからイベント通知の登録リストを取得
 *  \param[in]  type    通知タイプ
 *  \return     通知タイプに対応した登録リスト．無効な指定はnullptrが返る．
 */
/* ------------------------------------------------------------------------- */
RegisterList *Notifier::GetRegisterList(NotifyType type) noexcept
{
    if (type >= NotifyType::Reserve_Start)
    {
        return nullptr;
    }

    return &_registerListArray[static_cast<size_t>(type)];
}
}    // namespace MGL::Event
// vim: et ts=4 sw=4 sts=4
