// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_singleton.cc
 *  \brief      MGL シングルトンクラス
 *  \date       Since: July 17, 2022. 4:09:40 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <algorithm>

#include <mgl/common/mgl_singleton.h>

namespace MGL
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      シングルトン解放クラスの初期化
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool SingletonFinalizer::Initialize() noexcept
{
    try
    {
        _mutex = std::make_unique<std::recursive_mutex>();
    }
    catch (...)
    {
        return false;
    }

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      シングルトンクラスの登録
 *  \param[in]  function    解放処理の関数
 */
/* ------------------------------------------------------------------------- */
void SingletonFinalizer::Register(FinalizeFunction function) noexcept
{
    if (_mutex == nullptr)
    {
        Initialize();
    }

    const std::scoped_lock lock(*_mutex);

    _functions.push_back(function);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      登録されたシングルトンクラスの解放
 */
/* ------------------------------------------------------------------------- */
void SingletonFinalizer::Finalize() noexcept
{
    const std::scoped_lock lock(*_mutex);

    while (!_functions.empty())
    {
        _functions.back()();
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      シングルトンクラスの登録解除
 *  \param[in]  function    解放処理の関数
 */
/* ------------------------------------------------------------------------- */
void SingletonFinalizer::Unregister(FinalizeFunction function) noexcept
{
    const std::scoped_lock lock(*_mutex);

    const auto it = std::find(_functions.begin(), _functions.end(), function);
    if (it != _functions.end())
    {
        _functions.erase(it);
    }
}
}    // namespace MGL

// vim: et ts=4 sw=4 sts=4
