// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_byte_stream.cc
 *  \brief      バイトデータストリームクラス
 *  \date       Since: November 12, 2017. 2:49:47 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/common/mgl_byte_stream.h>

#include <cstring>
#include <random>

namespace MGL
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      バッファの設定
 *  \param[in]  address 設定するバッファのアドレス
 *  \param[in]  size    バッファのサイズ
 */
/* ------------------------------------------------------------------------- */
void ByteStream::SetBuffer(void *address, size_t size) noexcept
{
    _address = address;
    _size = size;
    _offset = 0;
    _isOverflowed = false;
    _isReadOnly = false;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      バッファの設定（読み込み専用）
 *  \param[in]  address 設定するバッファのアドレス
 *  \param[in]  size    バッファのサイズ
 */
/* ------------------------------------------------------------------------- */
void ByteStream::SetBuffer(const void *address, size_t size) noexcept
{
    _address = const_cast<void *>(address);
    _size = size;
    _offset = 0;
    _isOverflowed = false;
    _isReadOnly = true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      オフセットの設定
 *  \param[in]  offset  オフセット
 *  \retval     true    成功
 *  \retval     false   失敗（サイズオーバー）
 */
/* ------------------------------------------------------------------------- */
bool ByteStream::SetOffset(size_t offset) noexcept
{
    if (offset >= _size)
    {
        _isOverflowed = true;
        return false;
    }

    _isOverflowed = false;
    _offset = offset;

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      データのクリア
 *  \param[in]  clearType   初期化方式
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool ByteStream::Clear(ClearType clearType) noexcept
{
    if (_isReadOnly)
    {
        return false;
    }

    // ゼロ初期化
    if (clearType == ClearType::Zero)
    {
        memset(_address, 0, _size);
    }
    // ランダム初期化
    else if (clearType == ClearType::Random)
    {
        std::random_device rd;
        std::mt19937 mt(rd());

        uint32_t rand = 0;
        for (size_t i = 0; i < _size; i++)
        {
            if (rand == 0)
            {
                rand = static_cast<uint32_t>(mt());
            }
            else
            {
                rand = rand >> 8u;
            }

            auto *addr = static_cast<uint8_t *>(_address);
            addr[i] = rand & 0xFFu;
        }
    }

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      読み込み
 *  \param[out] dest    読み込む値の格納先
 *  \param[in]  size    読み込みサイズ
 *  \retval     true    成功
 *  \retval     false   失敗（サイズオーバー）
 */
/* ------------------------------------------------------------------------- */
bool ByteStream::Read(void *dest, size_t size) noexcept
{
    if (_isOverflowed)
    {
        return false;
    }

    if ((_offset + size) > _size)
    {
        _isOverflowed = true;
        return false;
    }

    uint8_t *addr = static_cast<uint8_t *>(_address) + _offset;
    memcpy(dest, addr, size);

    _offset += size;

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      書き込み
 *  \param[in]  source  書き込むデータの先頭アドレス
 *  \param[in]  size    書き込むデータのサイズ
 *  \retval     true    成功
 *  \retval     false   失敗（サイズオーバー）
 */
/* ------------------------------------------------------------------------- */
bool ByteStream::Write(const void *source, size_t size) noexcept
{
    if (_isReadOnly || _isOverflowed)
    {
        return false;
    }

    if ((_offset + size) > _size)
    {
        _isOverflowed = true;
        return false;
    }

    uint8_t *addr = static_cast<uint8_t *>(_address) + _offset;
    memcpy(addr, source, size);

    _offset += size;

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      論理値の読み込み
 *  \param[out] flag    読み込んだ論理値の格納先
 *  \retval     true    成功
 *  \retval     false   失敗（サイズオーバー）
 */
/* ------------------------------------------------------------------------- */
bool ByteStream::ReadBool(bool &flag) noexcept
{
    uint8_t temp = 0;
    if (!Read(temp))
    {
        return false;
    }

    flag = (temp != 0u);

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      論理値の読み込み
 *  \return     読み込んだ論理値．読み込み失敗時はfalse
 */
/* ------------------------------------------------------------------------- */
bool ByteStream::ReadBool() noexcept
{
    bool flag = false;
    ReadBool(flag);

    return flag;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      論理値の書き込み
 *  \param[in]  flag    書き込む論理値
 *  \retval     true    成功
 *  \retval     false   失敗（サイズオーバー）
 */
/* ------------------------------------------------------------------------- */
bool ByteStream::WriteBool(bool flag) noexcept
{
    if (_isReadOnly || _isOverflowed)
    {
        return false;
    }

    const uint8_t temp = flag ? 1 : 0;

    return Write(temp);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      バッファを任意の値で埋める
 *  \param[in]  value   書き込む値
 *  \param[in]  size    書き込みサイズ
 *  \retval     true    成功
 *  \retval     false   失敗（サイズオーバー）
 */
/* ------------------------------------------------------------------------- */
bool ByteStream::Fill(uint8_t value, size_t size) noexcept
{
    if (_isReadOnly || _isOverflowed)
    {
        return false;
    }

    if ((_offset + size) > _size)
    {
        _isOverflowed = true;
        return false;
    }

    uint8_t *addr = static_cast<uint8_t *>(_address) + _offset;
    memset(addr, value, size);

    _offset += size;

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ストリームのスキップ
 *  \param[in]  size    スキップするサイズ
 *  \retval     true    成功
 *  \retval     false   失敗（サイズオーバー）
 */
/* ------------------------------------------------------------------------- */
bool ByteStream::Skip(size_t size) noexcept
{
    if (_isOverflowed)
    {
        return false;
    }

    if ((_offset + size) > _size)
    {
        _isOverflowed = true;
        return false;
    }

    _offset += size;

    return true;
}
}    // namespace MGL

// vim: et ts=4 sw=4 sts=4
