// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_wave_voice.cc
 *  \brief      MGL WAVEボイス
 *  \date       Since: January 29, 2021. 6:34:50 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/audio/voice/mgl_audio_wave_voice.h>

#include <thread>

#include <mgl/audio/mgl_audio_player.h>
#include <mgl/audio/mgl_audio_sampletype_convert.h>
#include <mgl/file/mgl_file_utility.h>

namespace MGL::Audio
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  key         ボイスキー
 *  \param[in]  path        読み込むファイルのパス
 *  \param[in]  isLoop      ループフラグ
 *  \param[in]  loopFrame   ループ開始位置
 */
/* ------------------------------------------------------------------------- */
WaveVoice::WaveVoice(VoiceKey key, const File::PathView &path, bool isLoop, uint32_t loopFrame) noexcept
    : Voice(key, Type::Static)
    , _path(path)
    , _isLoop(isLoop)
    , _loopFrame(loopFrame)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボイスの読み込み処理
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool WaveVoice::Load() noexcept
{
    if (!File::Utility().Exists(_path))
    {
        return false;
    }

    SetStatus(Status::Loading);

    _future = std::async(std::launch::async, [this]
    {
        if (LoadWaveFile())
        {
            SetStatus(Status::Ready);
        }
        else
        {
            SetStatus(Status::Error);
        }
    });

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      WAVEファイルの読み込み
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool WaveVoice::LoadWaveFile() noexcept
{
    auto &player = Player::GetInstance();

    // ファイルをオープン
    WaveLoader wave;
    if (!wave.Open(_path))
    {
        player.Unload(GetKey());
        return false;
    }

    // フォーマットを取得
    _format = wave.GetFormat();

    // プレイヤーのサンプリングレートと一致していなければ失敗
    // 対処するならここにサンプリングレートの変換処理を入れる
    const auto &outputFormat = player.GetOutputFormat();
    if (static_cast<float>(_format.samplesPerSec) != outputFormat.samplesPerSec)
    {
        player.Unload(GetKey());
        return false;
    }

    // データ部の読み込み
    _dataSize = wave.GetDataSize();
    _data = STL::make_unique<std::byte[]>(_dataSize);
    _dataSize = wave.GetSample(_data.get(), _dataSize);

    // 最大フレーム数を算出
    _totalFrame = static_cast<uint32_t>(_dataSize) / _format.blockSize;

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      スタティックボイスのサンプルの取得
 *  \param[out] outDataL    左チャンネル出力の格納先
 *  \param[out] outDataR    右チャンネル出力の格納先
 *  \param[in]  trackIndex  トラック番号
 *  \param[in]  sampleFrame サンプルフレーム
 *  \retval     true        継続
 *  \retval     false       終了
 */
/* ------------------------------------------------------------------------- */
bool WaveVoice::GetSample(float &outDataL, float &outDataR, [[maybe_unused]] uint32_t trackIndex, size_t sampleFrame) const noexcept
{
    if (sampleFrame >= _totalFrame)
    {
        return false;
    }

    void *startPoint = _data.get() + (sampleFrame * _format.blockSize);

    SampleTypeConvert::ReadSample(&outDataL, &outDataR, startPoint, SampleType::SignedInt, _format.bitsPerSample, _format.channelCount);

    return (sampleFrame + 1) < _totalFrame;
}
}    // namespace MGL::Audio

// vim: et ts=4 sw=4 sts=4
