// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_wave_stream_voice.cc
 *  \brief      MGL WAVEストリームボイス
 *  \date       Since: January 30, 2021. 10:15:18 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/audio/voice/mgl_audio_wave_stream_voice.h>

#include <thread>

#include <mgl/audio/mgl_audio_player.h>
#include <mgl/file/mgl_file_utility.h>

namespace MGL::Audio
{
namespace
{
constexpr size_t kQueueBufferSize = static_cast<size_t>(64) * static_cast<size_t>(1024);
constexpr size_t kQueueBufferCount = 8;
}    // namespace

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  key         ボイスキー
 *  \param[in]  path        読み込むファイルのパス
 *  \param[in]  isLoop      ループフラグ
 *  \param[in]  loopFrame   ループ開始位置
 */
/* ------------------------------------------------------------------------- */
WaveStreamVoice::WaveStreamVoice(VoiceKey key, const File::PathView &path, bool isLoop, uint32_t loopFrame) noexcept
    : Voice(key, Voice::Type::Dynamic)
    , _path(path)
    , _loopFrame(loopFrame)
    , _streamQueue(kQueueBufferSize, kQueueBufferCount)
    , _isDefaultLoop(isLoop)
{
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デストラクタ
 */
/* ------------------------------------------------------------------------- */
WaveStreamVoice::~WaveStreamVoice() noexcept
{
    SetStatus(Status::Remove);
    _streamQueue.Reset();
    if (_readSampleThread.joinable())
    {
        _readSampleThread.join();
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボイスの読み込み処理
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool WaveStreamVoice::Load() noexcept
{
    // ファイルが存在しなければ失敗
    if (!File::Utility().Exists(_path))
    {
        return false;
    }

    // 読み込み中のステータスにして読み込みスレッドを起動
    SetStatus(Status::Loading);

    _loadFuture = std::async(std::launch::async, [this]
    {
        const std::scoped_lock lock(_mutex);

        if (LoadWaveFile())
        {
            SetStatus(Status::Ready);
        }
        else
        {
            SetStatus(Status::Error);
        }

        _condition.notify_one();
    });

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      WAVEファイルの読み込み
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool WaveStreamVoice::LoadWaveFile() noexcept
{
    auto &player = Player::GetInstance();

    // ファイルをオープン
    if (!_wave.Open(_path))
    {
        player.Unload(GetKey());
        return false;
    }

    // プレイヤーのサンプリングレートと一致していなければ失敗
    // 対処するならここにサンプリングレートの変換処理を入れる
    const auto &outputFormat = player.GetOutputFormat();
    if (static_cast<float>(_wave.GetFormat().samplesPerSec) != outputFormat.samplesPerSec)
    {
        player.Unload(GetKey());
        return false;
    }

    // 最大フレーム数を算出
    _totalFrame = static_cast<uint32_t>(_wave.GetDataSize()) / _wave.GetFormat().blockSize;

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボイスの開始処理
 *  \param[in]  trackIndex  トラック番号
 *  \param[in]  loopType    ループタイプ
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool WaveStreamVoice::Start([[maybe_unused]] uint32_t trackIndex, LoopType loopType) noexcept
{
    // 既に実行中であれば失敗
    if (_isBusy)
    {
        return false;
    }

    // 各種フラグを初期化
    _isBusy = true;
    _isFinished = false;
    _isLoop = ToLoopFlag(loopType, IsLoop(trackIndex));

    // サンプル読み込みスレッドを起動
    _readSampleThread = std::thread([this]
    {
        ReadSample();
        _isFinished = true;
    });

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボイスの再生終了
 */
/* ------------------------------------------------------------------------- */
void WaveStreamVoice::Stop() noexcept
{
    _isBusy = false;
    _streamQueue.Reset();
    if (_readSampleThread.joinable())
    {
        _readSampleThread.join();
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サンプルの読み込み
 */
/* ------------------------------------------------------------------------- */
void WaveStreamVoice::ReadSample() noexcept
{
    // ステータスが読み込み中であれば完了するまで待つ
    std::unique_lock lock(_mutex);
    _condition.wait(lock, [this]
    {
        return GetStatus() != Voice::Status::Loading;
    });

    if (GetStatus() != Voice::Status::Ready)
    {
        return;
    }

    _streamQueue.Reset();
    _wave.Seek(0);

    for (;;)
    {
        // ストリームキューに空きが発生するまで待機
        _streamQueue.Wait();
        if ((GetStatus() == Voice::Status::Remove) || !_isBusy)
        {
            break;
        }

        // ストリームキューから空きバッファを取得
        auto *const buffer = _streamQueue.GetFreeBuffer();
        if (buffer == nullptr)
        {
            continue;    // ここには来ないはず
        }

        // バッファにデータを格納してキューに追加
        buffer->channelCount = _wave.GetFormat().channelCount;
        buffer->dataSize = _wave.GetSampleWithConvert(buffer->data.get(), _streamQueue.GetBufferSize());
        _streamQueue.AddBuffer(buffer);

        // 読み込み元が終端に達していたらループまたは終了
        if (_wave.IsFinished())
        {
            if (_isLoop)
            {
                _wave.Seek(_loopFrame);
            }
            else
            {
                break;
            }
        }
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ダイナミックボイスのサンプルの取得
 *  \param[out] outDataL    左チャンネル出力の格納先
 *  \param[out] outDataR    右チャンネル出力の格納先
 *  \retval     true        継続
 *  \retval     false       終了
 */
/* ------------------------------------------------------------------------- */
bool WaveStreamVoice::GetSample(float &outDataL, float &outDataR) noexcept
{
    if (!_streamQueue.GetSample(outDataL, outDataR))
    {
        return !_isFinished;
    }

    return true;
}
}    // namespace MGL::Audio

// vim: et ts=4 sw=4 sts=4
