// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_wave_loader.cc
 *  \brief      MGL WAVEローダー
 *  \date       Since: January 27, 2021. 16:09:08 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/audio/voice/mgl_audio_wave_loader.h>

#include <mgl/audio/mgl_audio_sampletype_convert.h>
#include <mgl/file/mgl_file_exception.h>
#include <mgl/system/mgl_system_debug_macro.h>

namespace MGL::Audio
{
namespace
{
constexpr uint32_t kRiffID   = 0x46'46'49'52;   // RIFFチャンクID（"RIFF"）
constexpr uint32_t kWaveID   = 0x45'56'41'57;   // WAVEフォーマットID（"WAVE"）
constexpr uint32_t kFormatID = 0x20'74'6D'66;   // フォーマットチャンクID（"fmt "）
constexpr uint32_t kDataID   = 0x61'74'61'64;   // データチャンクID（"data"）
}    // namespace


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      WAVEファイルをオープン
 *  \param[in]  path    読み込むファイルのパス
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool WaveLoader::Open(const File::PathView &path) noexcept
{
    // 既に読み込み済みであれば失敗
    if (_isOpen)
    {
        return false;
    }

    try
    {
        // ファイルをオープン
        _file.Open(path);

        // RIFFチャンクかどうかをチェック
        uint32_t id = 0;
        _file.Read(&id, sizeof(id));
        if (id != kRiffID)
        {
            return false;
        }

        // RIFFチャンクのサイズは不要なのでスキップ
        _file.Skip(4);

        // WAVEフォーマットかどうかをチェック
        _file.Read(&id, sizeof(id));
        if (id != kWaveID)
        {
            return false;
        }

        // ファイルの終端までチャンクを読み込む
        while (!_file.IsEOF())
        {
            // チャンクIDとサイズを読み込み
            id = 0;
            _file.Read(&id, sizeof(id));
            uint32_t size = 0;
            _file.Read(&size, sizeof(size));

            // チャンクIDに応じて分岐
            switch (id)
            {
                // フォーマットチャンクならフォーマットを読み込み
                case kFormatID:
                    _file.Read(&_format, sizeof(_format));

                    // フォーマットチャンクのサイズが16バイトでない場合は拡張情報が入っているが，扱わないので飛ばす
                    if (size > sizeof(_format))
                    {
                        _file.Skip(size - sizeof(_format));
                    }
                    break;

                // データチャンクならそのオフセットを保持する
                case kDataID:
                    _dataOffset = _file.GetOffset();
                    _dataSize = size;
                    _file.Skip(size);
                    break;

                // それ以外のチャンクは読み取らない
                default:
                    _file.Skip(size);
                    break;
            }
        }
    }
    catch (const File::Exception &fileException)
    {
        MGL_ERROR(fileException.what());
        return false;
    }

    // PCM以外の形式には対応しない
    if (_format.type != 1)
    {
        // WaveLoader is only support PCM.
        return false;
    }

#if 0
    printf("[WaveLoader] %s:\n", path);
    printf("  type: %d\n", _format.type);
    printf("  channelCount: %d\n", _format.channelCount);
    printf("  samplesPerSec: %d\n", _format.samplesPerSec);
    printf("  bytesPerSec: %d\n", _format.bytesPerSec);
    printf("  blockSize: %d\n", _format.blockSize);
    printf("  bitsPerSample: %d\n", _format.bitsPerSample);
#endif

    _isOpen = true;

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      WAVEファイルをクローズ
 */
/* ------------------------------------------------------------------------- */
void WaveLoader::Close() noexcept
{
    if (IsOpen())
    {
        _file.Close();
        _isOpen = false;
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サンプルの取得
 *  \param[in]  buffer      サンプルの格納先
 *  \param[in]  bufferSize  サンプルの格納先のサイズ
 *  \return     実際に格納されたサイズ
 */
/* ------------------------------------------------------------------------- */
size_t WaveLoader::GetSample(void *buffer, size_t bufferSize) noexcept
{
    // オープン前は実行しない
    if (!IsOpen())
    {
        return 0;
    }

    // データの終端に達していたら読み込まない
    if (IsFinished())
    {
        return 0;
    }

    // 中途半端なサンプルを読み込まないよう要求サイズを調整する
    size_t requestSize = bufferSize;
    if (auto mod = requestSize % _format.blockSize; mod != 0)
    {
        requestSize -= mod;
    }

    // 読み込み
    try
    {
        _file.Seek(File::SeekType::Top, _dataOffset + _currentDataOffset);
        auto readSize = _file.Read(buffer, requestSize);
        _currentDataOffset += readSize;

        return readSize;
    }
    catch (const File::Exception &fileException)
    {
        MGL_ERROR(fileException.what());
        return 0;
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サンプルをfloatに変換して取得
 *  \param[in]  buffer      サンプルの格納先
 *  \param[in]  bufferSize  サンプルの格納先のサイズ
 *  \return     実際に格納されたサイズ
 */
/* ------------------------------------------------------------------------- */
size_t WaveLoader::GetSampleWithConvert(float *buffer, size_t bufferSize) noexcept
{
    // オープン前は実行しない
    if (!IsOpen())
    {
        return 0;
    }

    // データの終端に達していたら読み込まない
    if (IsFinished())
    {
        return 0;
    }

    // バッファの格納先のブロックサイズとフレーム数を算出
    auto destBlockSize = _format.channelCount * sizeof(float);
    auto destFrameCount = bufferSize / destBlockSize;

    // 一時バッファを準備
    auto requestSize = destFrameCount * _format.blockSize;
    auto readBuffer = STL::make_unique<std::byte[]>(requestSize);

    try
    {
        // ファイルから一時バッファに読み込み
        _file.Seek(File::SeekType::Top, _dataOffset + _currentDataOffset);
        auto readSize = _file.Read(readBuffer.get(), requestSize);
        _currentDataOffset += readSize;

        // 一時バッファから変換しつつfloatのバッファに格納
        void *top = readBuffer.get();
        for (size_t i = 0; i < destFrameCount; ++i)
        {
            auto *outDataL = buffer;
            auto *outDataR = (_format.channelCount >= 2) ? buffer + 1 : nullptr;

            top = SampleTypeConvert::ReadSample(outDataL, outDataR, top, SampleType::SignedInt, _format.bitsPerSample, _format.channelCount);

            buffer += _format.channelCount;
        }

        // 実際に読み込んだサイズを算出して返す
        return readSize / _format.blockSize * destBlockSize;
    }
    catch (const File::Exception &fileException)
    {
        MGL_ERROR(fileException.what());
        return 0;
    }
}
}    // namespace MGL::Audio

// vim: et ts=4 sw=4 sts=4
