// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_renderer_xaudio2.cc
 *  \brief      MGL XAudio2オーディオレンダラ
 *  \date       Since: April 2, 2021. 16:47:30 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/audio/renderer/mgl_audio_renderer_xaudio2.h>
#if defined(MGL_AUDIO_RENDERER_ENABLE_XAUDIO2)

#include <algorithm>
#include <cstring>

#if defined(min)
    #undef min
#endif

#include <mgl/system/mgl_system_debug_macro.h>

#pragma comment(lib, "xaudio2.lib")

namespace MGL::Audio
{
namespace
{
constexpr size_t kRingBufferSize = static_cast<size_t>(8) * static_cast<size_t>(1024);    //!< リングバッファ1つあたりのサイズ
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
XAudio2Renderer::XAudio2Renderer() noexcept
    : _isInitialize(false)
    , _xaudio2(nullptr)
    , _masteringVoice(nullptr)
    , _sourceVoice(nullptr)
    , _currentRingBuffer(0)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デストラクタ
 */
/* ------------------------------------------------------------------------- */
XAudio2Renderer::~XAudio2Renderer() noexcept
{
    if (_isInitialize)
    {
        // 途中でコールバック関数が呼ばれないよう先に停止しておく
        _sourceVoice->Stop();

        // ソースボイスを破棄
        if (_sourceVoice != nullptr)
        {
            _sourceVoice->DestroyVoice();
            _sourceVoice = nullptr;
        }

        // マスタリングボイスを破棄
        if (_masteringVoice != nullptr)
        {
            _masteringVoice->DestroyVoice();
            _masteringVoice = nullptr;
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \param[in]  initMode    初期化タイプ
 *  \param[in]  rendering   レンダリングを行う関数
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool XAudio2Renderer::Initialize(InitializeMode initMode, RenderingFunction rendering) noexcept
{
    // XAudio2本体を生成
    if (FAILED(XAudio2Create(&_xaudio2.p)))
    {
        MGL_WARNING("[MGL XAudio2] Failed to create XAudio2.");
        return false;
    }

    // レンダリングを行う関数を設定
    _rendering = rendering;

    // 要求タイプからサンプリングレートとチャンネル数を設定
    switch (initMode)
    {
        // 22KHz 1ch
        case InitializeMode::Sample22k1ch:
            _outputFormat.channelCount = 1;
            _outputFormat.samplesPerSec = 22050;
            break;

        // 22KHz 2ch
        case InitializeMode::Sample22k2ch:
            _outputFormat.channelCount = 2;
            _outputFormat.samplesPerSec = 22050;
            break;

        // 44KHz 1ch
        case InitializeMode::Sample44k1ch:
            _outputFormat.channelCount = 1;
            _outputFormat.samplesPerSec = 44100;
            break;

        // 44KHz 2ch
        case InitializeMode::Sample44k2ch:
            _outputFormat.channelCount = 2;
            _outputFormat.samplesPerSec = 44100;
            break;

        // デフォルト
        case InitializeMode::SystemDefault:
        default:
            _outputFormat.channelCount = XAUDIO2_DEFAULT_CHANNELS;
            _outputFormat.samplesPerSec = XAUDIO2_DEFAULT_SAMPLERATE;
            break;
    }
    _outputFormat.bitsPerSample = 16;
    _outputFormat.sampleType = SampleType::SignedInt;   // TODO
    _outputFormat.isInterleaved = false;

    // マスターボイスを生成
    if (FAILED(_xaudio2->CreateMasteringVoice(
        &_masteringVoice,
        static_cast<UINT32>(_outputFormat.channelCount),
        static_cast<UINT32>(_outputFormat.samplesPerSec)
    )))
    {
        MGL_WARNING("[MGL XAudio2] Failed to create master voice.");
        return false;
    }

    // ソースボイス用のフォーマットを生成
    WAVEFORMATEX waveFormat;
    waveFormat.cbSize = sizeof(WAVEFORMATEX);
    waveFormat.wFormatTag = WAVE_FORMAT_PCM;
    waveFormat.nChannels = static_cast<WORD>(_outputFormat.channelCount);
    waveFormat.nSamplesPerSec = static_cast<DWORD>(_outputFormat.samplesPerSec);
    waveFormat.wBitsPerSample = static_cast<WORD>(_outputFormat.bitsPerSample);
    waveFormat.nBlockAlign = waveFormat.nChannels * waveFormat.wBitsPerSample / 8;
    waveFormat.nAvgBytesPerSec = waveFormat.nSamplesPerSec * waveFormat.nBlockAlign;

    // ソースボイスを生成
    if (FAILED(_xaudio2->CreateSourceVoice(
        &_sourceVoice,
        &waveFormat,
        0,
        2.0f,
        this
    )))
    {
        MGL_WARNING("[MGL XAudio2] Failed to create source voice.");
        return false;
    }

    // リングバッファを準備
    for (auto &buffer : _ringBuffer)
    {
        buffer = STL::make_unique<std::byte[]>(kRingBufferSize);
    }
    _currentRingBuffer = 0;

    _sourceVoice->Start();

    _isInitialize = true;

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボイス再生開始時のコールバック関数
 *  \param[in]  samplesRequired   要求サンプルフレーム数
 *  \note
 *      継承元の変数名は"BytesRequired"だが，実際にはバイト数ではなくサンプル数が届く．
 *      MSDNのリファレンスでは"SamplesRequired"になっている．
 */
/* ------------------------------------------------------------------------- */
void STDMETHODCALLTYPE XAudio2Renderer::OnVoiceProcessingPassStart(UINT32 samplesRequired)
{
    // 要求がなければ実行しない
    if (samplesRequired == 0)
    {
        return;
    }

    // 要求フレームを算出
    // バッファのサイズより大きい場合は補正する
    uint32_t frameCount = samplesRequired;
    const auto frameMax = static_cast<uint32_t>((kRingBufferSize / (_outputFormat.bitsPerSample / 8) / _outputFormat.channelCount));
    frameCount = std::min(frameCount, frameMax);

    // バッファを準備
    void *buffer = reinterpret_cast<void *>(_ringBuffer[_currentRingBuffer++].get());
    if (_currentRingBuffer >= kRingBufferCount)
    {
        _currentRingBuffer = 0;
    }
    void *outData[1];
    outData[0] = buffer;

    // プレイヤーのレンダリング処理を実行
    _rendering(outData, 1, _outputFormat, frameCount);

    // ソースボイスにバッファを追加
    XAUDIO2_BUFFER submitBuffer = {};
    submitBuffer.AudioBytes = static_cast<UINT32>(frameCount * (_outputFormat.bitsPerSample / 8) * _outputFormat.channelCount);
    submitBuffer.pAudioData = reinterpret_cast<BYTE*>(buffer);
    auto hr = _sourceVoice->SubmitSourceBuffer(&submitBuffer);
    if (FAILED(hr))
    {
        MGL_WARNING("[MGL XAudio2] Failed to submit source buffer.");
    }
}
}   // namespace MGL::Audio

#endif  // MGL_AUDIO_RENDERER_ENABLE_XAUDIO2

// vim: et ts=4 sw=4 sts=4
