// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_renderer_apple_coreaudio.mm
 *  \brief      MGL Apple CoreAudio オーディオレンダラ
 *  \date       Since: January 17, 2021. 5:46:09 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/audio/renderer/mgl_audio_renderer_apple_coreaudio.h>
#if defined(MGL_AUDIO_RENDERER_ENABLE_APPLE_COREAUDIO)

#include <AVFoundation/AVFoundation.h>

namespace MGL::Audio
{
namespace
{
#if defined(MGL_TARGET_IOS)
constexpr int kResumeRetryCount = 20;   // オーディオ再開の試行回数
#endif
constexpr size_t kRenderingBufferCount = 16; // 再生に使用するバッファの最大数
}
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
AppleCoreAudioRenderer::AppleCoreAudioRenderer() noexcept
    : Renderer()
    , _isInitialize(false)
    , _eventSuspend(Event::NotifyType::AudioSuspend, OnEventSuspend, this)
    , _eventResume(Event::NotifyType::AudioResume, OnEventResume, this)
    , _outputFormatASBD()
    , _outUnit()
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デストラクタ
 */
/* ------------------------------------------------------------------------- */
AppleCoreAudioRenderer::~AppleCoreAudioRenderer() noexcept
{
    if (_isInitialize)
    {
        AudioOutputUnitStop(_outUnit);
        AudioUnitUninitialize(_outUnit);

        _isInitialize = false;
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \param[in]  initMode    初期化タイプ
 *  \param[in]  rendering   レンダリングを行う関数
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool AppleCoreAudioRenderer::Initialize(InitializeMode initMode, RenderingFunction rendering) noexcept
{
    OSStatus error;
    
    // レンダリングを行う関数を設定
    if (rendering == nullptr)
    {
        return false;
    }
    _rendering = rendering;
    
    // オーディオセッションの初期化
#if defined(MGL_TARGET_IOS) || defined(MGL_TARGET_TVOS)
    AVAudioSession *session = [AVAudioSession sharedInstance];
    [session setCategory:AVAudioSessionCategoryAmbient
                   error:nil];
    [session setActive:YES
                 error:nil];
#endif
    
    // オーディオユニットの取得
    AudioComponentDescription descriptor;
    descriptor.componentType = kAudioUnitType_Output;
#if defined(MGL_TARGET_MACOS)
    descriptor.componentSubType = kAudioUnitSubType_DefaultOutput;
#else
    descriptor.componentSubType = kAudioUnitSubType_RemoteIO;
#endif
    descriptor.componentManufacturer = kAudioUnitManufacturer_Apple;
    descriptor.componentFlags = 0;
    descriptor.componentFlagsMask = 0;

    AudioComponent component = AudioComponentFindNext(nullptr, &descriptor);
    if (AudioComponentInstanceNew(component, &_outUnit) != noErr)
    {
        return false;
    }

    // 出力フォーマットの取得と再設定
    UInt32 size = sizeof(_outputFormatASBD);
    error = AudioUnitGetProperty(_outUnit, kAudioUnitProperty_StreamFormat, kAudioUnitScope_Output, 0, &_outputFormatASBD, &size);
    if (error != noErr)
    {
        return false;
    }

    // 要求タイプから設定したいパラメータを書き換える
    switch (initMode)
    {
        // 44KHz 2ch
        case InitializeMode::Sample44k2ch:
            _outputFormatASBD.mSampleRate = 44100;
            _outputFormatASBD.mChannelsPerFrame = 2;
            break;

        // 22KHz 2ch
        case InitializeMode::Sample22k2ch:
            _outputFormatASBD.mSampleRate = 22050;
            _outputFormatASBD.mChannelsPerFrame = 2;
            break;

        // 1ch出力は非対応
        case InitializeMode::Sample44k1ch:
        case InitializeMode::Sample22k1ch:
            return false;

        case InitializeMode::SystemDefault:
        default:
            break;
    }
    error = AudioUnitSetProperty(_outUnit, kAudioUnitProperty_StreamFormat, kAudioUnitScope_Input, 0, &_outputFormatASBD, sizeof(_outputFormatASBD));
    if (error != noErr)
    {
        return false;
    }

    // コールバック関数の登録
    AURenderCallbackStruct input;
    input.inputProc = renderCallback;
    input.inputProcRefCon = this;

    error = AudioUnitSetProperty(
        _outUnit,
        kAudioUnitProperty_SetRenderCallback,
        kAudioUnitScope_Input,
        0,
        &input,
        sizeof(input)
    );
    if (error != noErr)
    {
        return false;
    }

    // AudioUnitの初期化
    if (AudioUnitInitialize(_outUnit) != noErr)
    {
        return false;
    }

    if (AudioOutputUnitStart(_outUnit) != noErr)
    {
        return false;
    }

    // 出力フォーマットを保存
    _outputFormat.samplesPerSec = static_cast<float>(_outputFormatASBD.mSampleRate);
    if ((_outputFormatASBD.mFormatFlags & kAudioFormatFlagIsSignedInteger) != 0)
    {
        _outputFormat.sampleType = SampleType::SignedInt;
    }
    else if ((_outputFormatASBD.mFormatFlags & kAudioFormatFlagIsFloat) != 0)
    {
        _outputFormat.sampleType = SampleType::Float;
    }
    _outputFormat.isInterleaved = ((_outputFormatASBD.mFormatFlags & kAudioFormatFlagIsNonInterleaved) == 0);
    _outputFormat.bitsPerSample = _outputFormatASBD.mBitsPerChannel;
    _outputFormat.channelCount = _outputFormatASBD.mChannelsPerFrame;

    _isInitialize = true;
    
    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          レンダリングコールバック
 *  \param[in]      inRefCon        このクラスのアドレス
 *  \param[in]      ioActionFlags   未使用
 *  \param[in]      inTimeStamp     未使用
 *  \param[in]      inBusNumber     未使用
 *  \param[in]      inNumberFrames  出力するフレーム数
 *  \param[in,out]  ioData          出力するオーディオデータの格納先
 *  \return         処理結果（常にエラーなし）
 */
/* ------------------------------------------------------------------------- */
OSStatus AppleCoreAudioRenderer::renderCallback(
    void *inRefCon,
    [[maybe_unused]] AudioUnitRenderActionFlags *ioActionFlags,
    [[maybe_unused]] const AudioTimeStamp *inTimeStamp,
    [[maybe_unused]] UInt32 inBusNumber,
    UInt32 inNumberFrames,
    AudioBufferList *ioData
) noexcept
{
    auto *thisPtr = static_cast<AppleCoreAudioRenderer *>(inRefCon);
    std::array<void *, kRenderingBufferCount> outData;
    
    // 自分自身の初期化が終わる前に呼ばれた場合は何もしない
    if (!thisPtr->_isInitialize)
    {
        return noErr;
    }

    const auto bufferCount = std::min(ioData->mNumberBuffers, static_cast<UInt32>(outData.size()));
    if (bufferCount >= 1)
    {
        // プレイヤー側のコールバックに渡す格納先バッファを設定
        for (size_t i = 0; i < ioData->mNumberBuffers; i++)
        {
            outData[i] = ioData->mBuffers[i].mData;
        }

        // プレイヤー側のコールバック関数に処理を渡す
        thisPtr->_rendering(&(outData.front()), ioData->mNumberBuffers, thisPtr->_outputFormat, inNumberFrames);
    }

    return noErr;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      オーディオの中断コールバック
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       未使用
 *  \note
 *      iOSで着信などの優先度の高いオーディオ出力が扱われた際，アプリケーション側のオーディオを
 *      停止させる必要があるため，その処理のためのコールバック関数．
 */
/* ------------------------------------------------------------------------- */
void AppleCoreAudioRenderer::OnEventSuspend(void *callbackArg, [[maybe_unused]] void *notifyArg) noexcept
{
#if defined(MGL_TARGET_IOS)
    auto *thisPtr = static_cast<AppleCoreAudioRenderer *>(callbackArg);
    
    if (thisPtr->_isInitialize)
    {
        [AVAudioSession.sharedInstance setActive:NO
                                           error:nil];
        
        AudioOutputUnitStop(thisPtr->_outUnit);
    }
#else
    (void)callbackArg;
#endif
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      オーディオの再開コールバック
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       未使用
 */
/* ------------------------------------------------------------------------- */
void AppleCoreAudioRenderer::OnEventResume(void *callbackArg, [[maybe_unused]] void *notifyArg) noexcept
{
#if defined(MGL_TARGET_IOS)
    auto *thisPtr = static_cast<AppleCoreAudioRenderer *>(callbackArg);
    
    if (thisPtr->_isInitialize)
    {
        auto *session = AVAudioSession.sharedInstance;
        
        [session setCategory:AVAudioSessionCategoryAmbient
                       error:nil];

        // 通知されてからアクティベート可能になるまでに若干のラグがあるらしく，直後に行うと稀に失敗するため，
        // アクティベートが完了するまで試行を繰り返す．
        // （もうちょっと良いやり方がありそうだが，復帰直後に行わないと正常に再開できないらしい）
        for (int tryCount = 0; tryCount < kResumeRetryCount; ++tryCount)
        {
            sleep(1);
            NSError *error;
            [session setActive:YES
                         error:&error];
            if (error == nil)
            {
                AudioOutputUnitStart(thisPtr->_outUnit);
                return;
            }
        }
        
        NSLog(@"[MGL-Audio] Warning: Failed to resume audio.");
    }
#else
    (void)callbackArg;
#endif
}
}   // namespace MGL::Audio

#endif  // MGL_AUDIO_RENDERER_ENABLE_APPLE_COREAUDIO
// vim: et ts=4 sw=4 sts=4
