// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_voice_storage.cc
 *  \brief      MGL オーディオボイスストレージ
 *  \date       Since: January 16, 2021. 2:47:20 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/audio/mgl_audio_voice_storage.h>

namespace MGL::Audio
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボイスの追加
 *  \param[in]  voice   追加するボイス
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool VoiceStorage::Add(const SharedVoice &voice) noexcept
{
    // NOTE: プレイヤー側がキーの重複を見ているため，ここでは単純に追加するだけ

    const std::scoped_lock lock(_mutex);

    _voices.emplace(voice->GetKey(), voice);

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボイスの取得
 *  \param[in]  key     ボイスキー
 *  \return     ボイスキーに対応したボイス
 */
/* ------------------------------------------------------------------------- */
SharedVoice VoiceStorage::Get(VoiceKey key) noexcept
{
    const std::scoped_lock lock(_mutex);

    const auto it = _voices.find(key);
    if (it == _voices.end())
    {
        return nullptr;
    }

    return it->second;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボイスの削除
 *  \param[in]  key     ボイスキー
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool VoiceStorage::Remove(VoiceKey key) noexcept
{
    const std::scoped_lock lock(_mutex);

    const auto it = _voices.find(key);
    if (it == _voices.end())
    {
        return false;
    }

    it->second->RemoveRequests();

    _voices.erase(it);

    return true;
}
}    // namespace MGL::Audio

// vim: et ts=4 sw=4 sts=4
