// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_stream_queue.cc
 *  \brief      MGL オーディオストリームキュー
 *  \date       Since: January 29, 2021. 17:20:39 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/audio/mgl_audio_stream_queue.h>

namespace MGL::Audio
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  bufferSize  バッファサイズ
 *  \param[in]  bufferCount バッファの数
 */
/* ------------------------------------------------------------------------- */
StreamQueue::StreamQueue(size_t bufferSize, uint32_t bufferCount) noexcept
    : _bufferArray(STL::make_unique<QueueBuffer[]>(bufferCount))
    , _bufferSize(bufferSize)
    , _bufferCount(bufferCount)
{
    for (size_t i = 0; i < bufferCount; ++i)
    {
        _bufferArray[i].data = STL::make_unique<float[]>(bufferSize / sizeof(float));
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ストリームキューを初期状態に戻す
 */
/* ------------------------------------------------------------------------- */
void StreamQueue::Reset() noexcept
{
    const std::scoped_lock lock(_mutex);

    _top = _tail = nullptr;
    for (size_t i = 0; i < _bufferCount; i++)
    {
        _bufferArray[i].isActive = false;
        _bufferArray[i].next = nullptr;
    }

    _activeBufferCount = 0;
    _currentFrameOffset = 0;

    _condition.notify_one();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      空いているバッファを取得
 *  \return     空いているバッファ．空きがない場合はnullptr
 */
/* ------------------------------------------------------------------------- */
StreamQueue::QueueBuffer *StreamQueue::GetFreeBuffer() noexcept
{
    const std::scoped_lock lock(_mutex);

    for (size_t i = 0; i < _bufferCount; ++i)
    {
        if (!_bufferArray[i].isActive)
        {
            return &_bufferArray[i];
        }
    }

    return nullptr;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      バッファを追加
 *  \param[in]  queueBuffer     追加するバッファ
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool StreamQueue::AddBuffer(StreamQueue::QueueBuffer *queueBuffer) noexcept
{
    const std::scoped_lock lock(_mutex);

    // モノラル・ステレオのどちらかでなければ追加しない
    if ((queueBuffer->channelCount != 1) && (queueBuffer->channelCount != 2))
    {
        return false;
    }

    // キューが空なら先頭に追加
    if (_top == nullptr)
    {
        _top = _tail = queueBuffer;
    }
    // キューが空でなければ末尾に追加
    else
    {
        _tail->next = queueBuffer;
        _tail = queueBuffer;
    }

    queueBuffer->next = nullptr;
    queueBuffer->isActive = true;

    _activeBufferCount++;

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サンプルデータの取得
 *  \param[out] frameDataL      左チャンネル出力の格納先
 *  \param[out] frameDataR      右チャンネル出力の格納先
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool StreamQueue::GetSample(float &frameDataL, float &frameDataR) noexcept
{
    const std::scoped_lock lock(_mutex);

    // キューが空なら何もしない
    if (_top == nullptr)
    {
        return false;
    }

    // モノラル音声なら両方の出力に同じサンプルを設定
    if (_top->channelCount == 1)
    {
        frameDataL = frameDataR = _top->data[_currentFrameOffset];
        _currentFrameOffset++;
    }
    // ステレオ音声ならそれぞれにサンプルを設定
    else if (_top->channelCount == 2)
    {
        frameDataL = _top->data[_currentFrameOffset];
        frameDataR = _top->data[static_cast<size_t>(_currentFrameOffset) + 1];
        _currentFrameOffset += 2;
    }
    else
    {
        return false;
    }

    // 先頭にあるバッファの読み込みが終端に達したらそのバッファを解放して次のバッファを設定する
    if (_currentFrameOffset >= (_top->dataSize / sizeof(float)))
    {
        _top->isActive = false;
        _top = _top->next;

        if (_top == nullptr)
        {
            _tail = nullptr;
        }

        _currentFrameOffset = 0;

        // 空き待ちのスレッドをここで起床させる
        _activeBufferCount--;
        _condition.notify_one();
    }

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      キューが操作可能になるまで待つ
 */
/* ------------------------------------------------------------------------- */
void StreamQueue::Wait() noexcept
{
    std::unique_lock lock(_mutex);

    _condition.wait(lock, [this]
    {
        return (_activeBufferCount < _bufferCount);
    });
}
}    // namespace MGL::Audio

// vim: et ts=4 sw=4 sts=4
