// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_source_instance.cc
 *  \brief      MGL オーディオソースインスタンス
 *  \date       Since: January 16, 2021. 3:33:45 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/audio/mgl_audio_source_instance.h>

namespace MGL::Audio
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
SourceInstance::SourceInstance(SharedVoice &voice, bool isAutoRemove) noexcept
    : _voice(voice)
    , _isAutoRemove(isAutoRemove)
{
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      再生開始
 *  \param[in]  trackIndex  トラック番号
 *  \param[in]  loopType    ループタイプ
 *  \param[in]  volume      音量
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool SourceInstance::Play(uint32_t trackIndex, LoopType loopType, float volume) noexcept
{
    const std::scoped_lock lock(_mutex);

    // 削除要求後は再生しない
    if (_status == Status::Remove)
    {
        return false;
    }

    // トラック番号がボイスのトラック数より大きい場合は失敗
    // ただし読み込み中はトラック数が不明なのでここではチェックしない
    if (_voice->GetStatus() == Voice::Status::Ready)
    {
        if (trackIndex >= _voice->GetTrackCount())
        {
            return false;
        }
    }

    // 既に再生中である場合は一旦ボイスに停止を通知する
    if (_status == Status::Play)
    {
        _voice->Stop();
    }

    // ボイスに再生開始を通知する
    if (!_voice->Start(trackIndex, loopType))
    {
        return false;
    }

    // パラメータを初期化して再生開始
    _trackIndex = trackIndex;
    _currentFrame = 0;
    _isPaused = false;
    _fade.Cancel();
    _loopType = loopType;
    _volume = volume;
    _status = Status::Play;

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ソースの更新処理
 *  \param[out] outDataL    左チャンネルの出力
 *  \param[out] outDataR    右チャンネルの出力
 *  \retval     true        再生継続
 *  \retval     false       再生終了
 */
/* ------------------------------------------------------------------------- */
bool SourceInstance::Update(float &outDataL, float &outDataR) noexcept
{
    const std::scoped_lock lock(_mutex);

    const auto voiceStatus = _voice->GetStatus();

    // ボイスの読み込みが完了するまで待機
    if (voiceStatus == Voice::Status::Loading)
    {
        return true;
    }

    // ボイスが無効になっている場合は再生終了
    if (voiceStatus == Voice::Status::Error)
    {
        return false;
    }

    // スタンバイ状態なら再生処理は行わない
    if (_status == Status::Standby)
    {
        return true;
    }

    // ボイスに削除要求がある場合はソースにも停止を要求する
    if (voiceStatus == Voice::Status::Remove)
    {
        // ポーズ中であれば即時削除
        if (_isPaused)
        {
            return false;
        }
        else
        {
            _status = Status::Remove;
        }
    }

    // ポーズ中は更新しない
    if (_isPaused)
    {
        return true;
    }

    // ボリュームはアトミックであり，内部で頻繁に読み書きするため，一旦ローカルに保持しておく
    float volume = _volume;

    // 停止要求があったら短時間でフェードアウトさせる．一般的なぶつ切りノイズ対策
    if (_status == Status::Remove)
    {
        volume -= 0.1f;
        if (volume <= 0.0f)
        {
            _volume = volume;
            return false;
        }
    }

    // ボイスからサンプルを取得
    bool isContinued = true;
    if (_voice->GetType() == Voice::Type::Static)
    {
        // スタティックボイスはサンプル取得後，現在のフレームをソース側で更新する必要がある
        if (!_voice->GetSample(outDataL, outDataR, _trackIndex, _currentFrame++))
        {
            if (ToLoopFlag(_loopType, _voice->IsLoop(_trackIndex)))
            {
                _currentFrame = _voice->GetLoopFrame(_trackIndex);
            }
            else
            {
                isContinued = false;
            }
        }
    }
    else if (_voice->GetType() == Voice::Type::Dynamic)
    {
        isContinued = _voice->GetSample(outDataL, outDataR);
    }

    // フェードの更新
    if (_fade.IsActive())
    {
        if (!_fade.Update(volume))
        {
            if (_fade.IsAutoStop())
            {
                isContinued = false;
            }
        }
    }

    // ボリュームを適用してメンバに書き戻す
    auto voiceVolume = _voice->GetVolume();
    outDataL *= volume * voiceVolume;
    outDataR *= volume * voiceVolume;
    _volume = volume;

    if (!isContinued)
    {
        if (_isAutoRemove)
        {
            return false;
        }
        else
        {
            _voice->Stop();
            _status = Status::Standby;
        }
    }

    return true;
}
}    // namespace MGL::Audio

// vim: et ts=4 sw=4 sts=4
