// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_sampletype_convert.cc
 *  \brief      MGL オーディオサンプルの変換用関数
 *  \date       Since: January 17, 2021. 6:47:12 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/audio/mgl_audio_sampletype_convert.h>

#include <cstdint>

namespace MGL::Audio::SampleTypeConvert
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief          符号付き8bit整数データに書き込み
 *  \param[in,out]  outDataL    左チャンネルの書き込み開始位置．書き込み後にアドレスを更新．
 *  \param[in,out]  outDataR    右チャンネルの書き込み開始位置．書き込み後にアドレスを更新．
 *  \param[in]      dataFormat  書き込み先のデータフォーマット
 *  \param[in]      inDataL     左チャンネルへ書き込む値
 *  \param[in]      inDataR     右チャンネルへ書き込む値
 */
/* ------------------------------------------------------------------------- */
void WriteDataToSigned8(void **outDataL, void **outDataR, const DataFormat &dataFormat, float inDataL, float inDataR) noexcept
{
    auto *L = static_cast<int8_t *>(*outDataL);
    auto *R = static_cast<int8_t *>(*outDataR);
    if (dataFormat.channelCount == 1)
    {
        *L++ = static_cast<int8_t>(inDataL * 127.0f);
    }
    else
    {
        if (R == nullptr)
        {
            *L++ = static_cast<int8_t>(inDataL * 127.0f);
            *L++ = static_cast<int8_t>(inDataR * 127.0f);
        }
        else
        {
            *L++ = static_cast<int8_t>(inDataL * 127.0f);
            *R++ = static_cast<int8_t>(inDataR * 127.0f);
        }
    }

    *outDataL = L;
    *outDataR = R;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief          符号付き16bit整数データに書き込み
 *  \param[in,out]  outDataL    左チャンネルの書き込み開始位置．書き込み後にアドレスを更新．
 *  \param[in,out]  outDataR    右チャンネルの書き込み開始位置．書き込み後にアドレスを更新．
 *  \param[in]      dataFormat  書き込み先のデータフォーマット
 *  \param[in]      inDataL     左チャンネルへ書き込む値
 *  \param[in]      inDataR     右チャンネルへ書き込む値
 */
/* ------------------------------------------------------------------------- */
void WriteDataToSigned16(void **outDataL, void **outDataR, const DataFormat &dataFormat, float inDataL, float inDataR) noexcept
{
    auto *L = static_cast<int16_t *>(*outDataL);
    auto *R = static_cast<int16_t *>(*outDataR);
    if (dataFormat.channelCount == 1)
    {
        *L++ = static_cast<int16_t>(inDataL * 32767.0f);
    }
    else
    {
        if (R == nullptr)
        {
            *L++ = static_cast<int16_t>(inDataL * 32767.0f);
            *L++ = static_cast<int16_t>(inDataR * 32767.0f);
        }
        else
        {
            *L++ = static_cast<int16_t>(inDataL * 32767.0f);
            *R++ = static_cast<int16_t>(inDataR * 32767.0f);
        }
    }

    *outDataL = L;
    *outDataR = R;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief          符号なし8bit整数データに書き込み
 *  \param[in,out]  outDataL    左チャンネルの書き込み開始位置．書き込み後にアドレスを更新．
 *  \param[in,out]  outDataR    右チャンネルの書き込み開始位置．書き込み後にアドレスを更新．
 *  \param[in]      dataFormat  書き込み先のデータフォーマット
 *  \param[in]      inDataL     左チャンネルへ書き込む値
 *  \param[in]      inDataR     右チャンネルへ書き込む値
 */
/* ------------------------------------------------------------------------- */
void WriteDataToUnsigned8(void **outDataL, void **outDataR, const DataFormat &dataFormat, float inDataL, float inDataR) noexcept
{
    auto *L = static_cast<uint8_t *>(*outDataL);
    auto *R = static_cast<uint8_t *>(*outDataR);
    if (dataFormat.channelCount == 1)
    {
        *L++ = static_cast<uint8_t>((inDataL + 1.0f) / 2.0f * 255.0f);
    }
    else
    {
        if (R == nullptr)
        {
            *L++ = static_cast<uint8_t>((inDataL + 1.0f) * 255.0f);
            *L++ = static_cast<uint8_t>((inDataR + 1.0f) * 255.0f);
        }
        else
        {
            *L++ = static_cast<uint8_t>((inDataL + 1.0f) * 255.0f);
            *R++ = static_cast<uint8_t>((inDataR + 1.0f) * 255.0f);
        }
    }

    *outDataL = L;
    *outDataR = R;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief          符号なし16bit整数データに書き込み
 *  \param[in,out]  outDataL    左チャンネルの書き込み開始位置．書き込み後にアドレスを更新．
 *  \param[in,out]  outDataR    右チャンネルの書き込み開始位置．書き込み後にアドレスを更新．
 *  \param[in]      dataFormat  書き込み先のデータフォーマット
 *  \param[in]      inDataL     左チャンネルへ書き込む値
 *  \param[in]      inDataR     右チャンネルへ書き込む値
 */
/* ------------------------------------------------------------------------- */
void WriteDataToUnsigned16(void **outDataL, void **outDataR, const DataFormat &dataFormat, float inDataL, float inDataR) noexcept
{
    auto *L = static_cast<uint16_t *>(*outDataL);
    auto *R = static_cast<uint16_t *>(*outDataR);
    if (dataFormat.channelCount == 1)
    {
        *L++ = static_cast<uint16_t>((inDataL + 1.0f) / 2.0f * 65535.0f);
    }
    else
    {
        if (R == nullptr)
        {
            *L++ = static_cast<uint16_t>((inDataL + 1.0f) * 65535.0f);
            *L++ = static_cast<uint16_t>((inDataR + 1.0f) * 65535.0f);
        }
        else
        {
            *L++ = static_cast<uint16_t>((inDataL + 1.0f) * 65535.0f);
            *R++ = static_cast<uint16_t>((inDataR + 1.0f) * 65535.0f);
        }
    }

    *outDataL = L;
    *outDataR = R;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief          32bit浮動小数点数データに書き込み
 *  \param[in,out]  outDataL    左チャンネルの書き込み開始位置．書き込み後にアドレスを更新．
 *  \param[in,out]  outDataR    右チャンネルの書き込み開始位置．書き込み後にアドレスを更新．
 *  \param[in]      dataFormat  書き込み先のデータフォーマット
 *  \param[in]      inDataL     左チャンネルへ書き込む値
 *  \param[in]      inDataR     右チャンネルへ書き込む値
 */
/* ------------------------------------------------------------------------- */
void WriteDataToFloat(void **outDataL, void **outDataR, const DataFormat &dataFormat, float inDataL, float inDataR) noexcept
{
    auto *L = static_cast<float *>(*outDataL);
    auto *R = static_cast<float *>(*outDataR);
    if (dataFormat.channelCount == 1)
    {
        *L++ = inDataL;
    }
    else
    {
        if (R == nullptr)
        {
            *L++ = inDataL;
            *L++ = inDataR;
        }
        else
        {
            *L++ = inDataL;
            *R++ = inDataR;
        }
    }

    *outDataL = L;
    *outDataR = R;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      データ書き込み用関数を取得
 *  \param[in]  dataFormat  書き込み先のデータフォーマット
 *  \return     対応した書き込み用関数のアドレス
 */
/* ------------------------------------------------------------------------- */
WriteDataFunction GetWriteDataFunction(const DataFormat &dataFormat) noexcept
{
    // 符号付き整数
    if (dataFormat.sampleType == SampleType::SignedInt)
    {
        // 8bit
        if (dataFormat.bitsPerSample == 8)
        {
            return WriteDataToSigned8;
        }
        // 16bit
        else if (dataFormat.bitsPerSample == 16)
        {
            return WriteDataToSigned16;
        }
    }
    // 符号なし整数
    else if (dataFormat.sampleType == SampleType::UnsignedInt)
    {
        // 8bit
        if (dataFormat.bitsPerSample == 8)
        {
            return WriteDataToUnsigned8;
        }
        // 16bit
        else if (dataFormat.bitsPerSample == 16)
        {
            return WriteDataToUnsigned16;
        }
    }
    // 浮動小数点数
    else if (dataFormat.sampleType == SampleType::Float)
    {
        // 32bit
        if (dataFormat.bitsPerSample == 32)
        {
            return WriteDataToFloat;
        }
    }

    return nullptr;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サンプルの読み込み
 *  \param[out] outDataL        左チャンネル or モノラル音声の格納先アドレス
 *  \param[out] outDataR        右チャンネルの格納先アドレス（nullable）
 *  \param[in]  inData          読み込みアドレス
 *  \param[in]  sampleType      読み込み元のサンプルタイプ
 *  \param[in]  bitPerSample    サンプルデータのサイズ（bit）
 *  \param[in]  channelCount    読み込み元のチャンネル数
 *  \return     次の読み込み先（inDataから読み込んだサイズだけずらしたアドレス）
 */
/* ------------------------------------------------------------------------- */
void *ReadSample(float *outDataL,
                 float *outDataR,
                 void *inData,
                 SampleType sampleType,
                 int bitPerSample,
                 int channelCount)
{
    void *nextData = nullptr;
    *outDataL = 0.0f;
    if (outDataR != nullptr)
    {
        *outDataR = 0.0f;
    }

    // 8bit符号付き整数
    if ((sampleType == SampleType::SignedInt) && (bitPerSample == 8))
    {
        auto *data = static_cast<int8_t *>(inData);
        *outDataL = static_cast<float>(*data++) / 128.0f;
        if (outDataR != nullptr)
        {
            if (channelCount >= 2)
            {
                *outDataR = static_cast<float>(*data++) / 128.0f;
            }
            else
            {
                *outDataR = *outDataL;
            }
        }
        nextData = data;
    }
    // 16bit符号付き整数
    else if ((sampleType == SampleType::SignedInt) && (bitPerSample == 16))
    {
        auto *data = static_cast<int16_t *>(inData);
        *outDataL = static_cast<float>(*data++) / 32768.0f;
        if (outDataR != nullptr)
        {
            if (channelCount >= 2)
            {
                *outDataR = static_cast<float>(*data++) / 32768.0f;
            }
            else
            {
                *outDataR = *outDataL;
            }
        }
        nextData = data;
    }
    // 8bit符号なし整数
    else if ((sampleType == SampleType::UnsignedInt) && (bitPerSample == 8))
    {
        auto *data = static_cast<uint8_t *>(inData);
        *outDataL = (static_cast<float>(*data++) / 255.0f * 2.0f) - 1.0f;
        if (outDataR != nullptr)
        {
            if (channelCount >= 2)
            {
                *outDataR = (static_cast<float>(*data++) / 255.0f * 2.0f) - 1.0f;
            }
            else
            {
                *outDataR = *outDataL;
            }
        }
        nextData = data;
    }
    // 16bit符号なし整数
    else if ((sampleType == SampleType::UnsignedInt) && (bitPerSample == 16))
    {
        auto *data = static_cast<uint16_t *>(inData);
        *outDataL = (static_cast<float>(*data++) / 65535.0f * 2.0f) - 1.0f;
        if (outDataR != nullptr)
        {
            if (channelCount >= 2)
            {
                *outDataR = (static_cast<float>(*data++) / 65535.0f * 2.0f) - 1.0f;
            }
            else
            {
                *outDataR = *outDataL;
            }
        }
        nextData = data;
    }
    // 32bit浮動小数点数
    else if ((sampleType == SampleType::Float) && (bitPerSample == 32))
    {
        auto *data = static_cast<float *>(inData);
        *outDataL = *data++;
        if (outDataR != nullptr)
        {
            if (channelCount >= 2)
            {
                *outDataR = *data++;
            }
            else
            {
                *outDataR = *outDataL;
            }
        }
        nextData = data;
    }

    return nextData;
}
}    // namespace MGL::Audio::SampleTypeConvert

// vim: et ts=4 sw=4 sts=4
