// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_player.cc
 *  \brief      MGL オーディオプレイヤー
 *  \date       Since: January 16, 2021. 3:53:49 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/audio/mgl_audio_player.h>

#include <algorithm>

namespace MGL::Audio
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インスタンスの取得
 *  \return     インスタンスの参照
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Player> &Player::GetInstanceRef() noexcept
{
    static STL::unique_ptr<Player> sInstance = nullptr;
    return sInstance;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \param[in]  renderer    オーディオレンダラ
 *  \param[in]  initMode    初期化モード
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Player::Initialize(STL::unique_ptr<Renderer> &renderer, InitializeMode initMode) noexcept
{
    // 多重初期化禁止
    if (_isInitialized)
    {
        return false;
    }

    // 初期化モードがNoneならオーディオを使用しないということなので何もせずにtrueを返す
    if (initMode == InitializeMode::None)
    {
        return true;
    }

    // レンダラを設定
    if (renderer == nullptr)
    {
        return false;
    }
    _renderer = std::move(renderer);

    // レンダラを初期化
    if (!_renderer->Initialize(initMode, Rendering))
    {
        return false;
    }

    // データ書き込み用関数を設定
    _writeData = SampleTypeConvert::GetWriteDataFunction(_renderer->GetOutputFormat());
    if (_writeData == nullptr)
    {
        _renderer.reset();
        return false;
    }

    _isInitialized = true;

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボイスの読み込み
 *  \param[in]  voice   ボイス
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool Player::Load(const SharedVoice &voice) noexcept
{
    // 初期化前は常に失敗
    if (!_isInitialized)
    {
        return false;
    }

    // 受け取ったボイスが不正なら失敗
    if (voice == nullptr)
    {
        return false;
    }

    // 既にボイスストレージに登録されていれば何もしない（失敗にはしない）
    if (_voiceStorage.Get(voice->GetKey()) != nullptr)
    {
        return true;
    }

    // ボイスに対して読み込みを要求
    if (!voice->Load())
    {
        return false;
    }

    // ボイスを登録
    _voiceStorage.Add(voice);

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボイスを解放
 *  \param[in]  key     ボイスキー
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool Player::Unload(VoiceKey key) noexcept
{
    // 初期化前は常に失敗
    if (!_isInitialized)
    {
        return false;
    }

    return _voiceStorage.Remove(key);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ソースインスタンスを生成
 *  \param[in]  key             ボイスキー
 *  \param[in]  isAutoRemove    自動削除フラグ
 *  \return     生成したソースインスタンス．失敗時はnullptr
 */
/* ------------------------------------------------------------------------- */
SharedSourceInstance Player::MakeSourceInstance(VoiceKey key, bool isAutoRemove) noexcept
{
    // 初期化前は常に失敗
    if (!_isInitialized)
    {
        return nullptr;
    }

    // ボイスを取得
    auto voice = _voiceStorage.Get(key);
    if (voice == nullptr)
    {
        return nullptr;
    }

    // エラーが発生しているボイスは再生しない
    if (voice->GetStatus() == Voice::Status::Error)
    {
        return nullptr;
    }

    // ソースを生成して追加
    auto source = STL::make_shared<SourceInstance>(voice, isAutoRemove);
    {
        const std::scoped_lock lock(_mutex);
        _sources.push_back(source);
    }

    return source;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボイスのボリュームを設定
 *  \param[in]  voiceKey        ボイスキー
 *  \param[in]  volume          ボリューム
 */
/* ------------------------------------------------------------------------- */
void Player::SetVoiceVolume(VoiceKey voiceKey, float volume) noexcept
{
    if (auto voice = _voiceStorage.Get(voiceKey); voice != nullptr)
    {
        voice->SetVolume(volume);
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボイスのボリュームを設定
 *  \param[in]  voiceKey        ボイスキー
 *  \return     ボイスのボリューム．無効なボイスキーを指定した場合は0.0f
 */
/* ------------------------------------------------------------------------- */
float Player::GetVoiceVolume(VoiceKey voiceKey) noexcept
{
    if (auto voice = _voiceStorage.Get(voiceKey); voice != nullptr)
    {
        return voice->GetVolume();
    }

    return 0.0f;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      レンダリング処理
 *  \param[out] outData         データの格納先
 *  \param[in]  outDataCount    データの格納先の数
 *  \param[in]  dataFormat      データフォーマット
 *  \param[in]  frameCount      レンダリングフレーム数
 */
/* ------------------------------------------------------------------------- */
void Player::Rendering(void **outData, size_t outDataCount, const DataFormat &dataFormat, size_t frameCount) noexcept
{
    if (!HasInstance())
    {
        return;
    }

    auto &player = GetInstance();

    void *outDataL = outData[0];
    void *outDataR = (outDataCount >= 2) ? outData[1] : nullptr;

    for (size_t i = 0; i < frameCount; i++)
    {
        float dataL = 0.0f;
        float dataR = 0.0f;

        {
            const std::scoped_lock lock(player._mutex);

            for (auto it = player._sources.begin(); it != player._sources.end();)
            {
                float sourceDataL = 0.0f;
                float sourceDataR = 0.0f;

                if (!(*it)->Update(sourceDataL, sourceDataR))
                {
                    it = player._sources.erase(it);
                }
                else
                {
                    ++it;
                }

                dataL += sourceDataL;
                dataR += sourceDataR;
            }
        }

        const float masterVolume = player._masterVolume;
        dataL = std::clamp(dataL * masterVolume, -1.0f, 1.0f);
        dataR = std::clamp(dataR * masterVolume, -1.0f, 1.0f);

        player._writeData(&outDataL, &outDataR, dataFormat, dataL, dataR);
    }
}
}    // namespace MGL::Audio

// vim: et ts=4 sw=4 sts=4
