// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_fade.cc
 *  \brief      MGL フェード処理
 *  \date       Since: January 31, 2021. 16:00:53 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/audio/mgl_audio_fade.h>

#include <cmath>

namespace MGL::Audio
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      開始
 *  \param[in]  currentVolume   現在の音量
 *  \param[in]  targetVolume    最終的な音量
 *  \param[in]  fadeTimeSec     フェード時間（秒）
 *  \param[in]  samplesPerSec   サンプリングレート
 *  \param[in]  isAutoStop      自動停止フラグ
 */
/* ------------------------------------------------------------------------- */
void Fade::Start(float currentVolume, float targetVolume, float fadeTimeSec, float samplesPerSec, bool isAutoStop) noexcept
{
    const std::scoped_lock lock(_mutex);

    _isActive = true;
    _isAutoStop = isAutoStop;
    _targetVolume = targetVolume;

    const auto fadeSampleLength = fadeTimeSec * samplesPerSec;
    _addVolumePerFrame = (targetVolume - currentVolume) / fadeSampleLength;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief          更新処理
 *  \param[in,out]  volume  変化させる音量
 *  \return         更新後の有効フラグ
 */
/* ------------------------------------------------------------------------- */
bool Fade::Update(float &volume) noexcept
{
    const std::scoped_lock lock(_mutex);

    // 加算量がゼロなら即終了
    if (std::fabs(_addVolumePerFrame) == 0.0f)
    {
        _isActive = false;
    }
    else
    {
        // 加算
        volume += _addVolumePerFrame;

        // 終了判定
        if ((_addVolumePerFrame > 0.0f) && (volume >= _targetVolume))
        {
            volume = _targetVolume;
            _isActive = false;
        }
        else if ((_addVolumePerFrame < 0.0f) && (volume <= _targetVolume))
        {
            volume = _targetVolume;
            _isActive = false;
        }
    }

    return _isActive;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      キャンセル
 */
/* ------------------------------------------------------------------------- */
void Fade::Cancel() noexcept
{
    const std::scoped_lock lock(_mutex);

    _addVolumePerFrame = 0.0f;
    _isActive = false;
}
}    // namespace MGL::Audio

// vim: et ts=4 sw=4 sts=4
