// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_achievement_server.cc
 *  \brief      MGL 実績サーバ
 *  \date       Since: September 8, 2021. 12:26:40 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/achievement/mgl_achievement_server.h>

namespace MGL::Achievement
{
namespace
{
STL::unique_ptr<Server> sInstance = nullptr;
constexpr size_t kTemporaryDataReserveSize = 64;    //!< テンポラリデータの予約サイズ
}    // namespace


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インスタンスの取得
 *  \return     インスタンスの参照
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Server> &Server::GetInstanceRef() noexcept
{
    return sInstance;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
Server::Server() noexcept
    : _eventPreFrameUpdate(Event::NotifyType::PreFrameUpdate, OnPreFrameUpdate, this)
{
    _temporaryDataArray.reserve(kTemporaryDataReserveSize);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \param[in]  delegate    デリゲート
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Server::Initialize(STL::unique_ptr<ServerDelegate> &delegate) noexcept
{
    if (_delegate != nullptr)
    {
        return false;
    }

    _delegate = std::move(delegate);

    return _delegate->Initialize();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      値を設定
 *  \param[in]  identifier  実績の識別子
 *  \param[in]  value       設定する値
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Server::SetValue(Identifier identifier, uint32_t value) noexcept
{
    if (auto *data = _delegate->GetAchievementData(identifier); data == nullptr)
    {
        return false;
    }

    _temporaryDataArray.emplace_back(identifier, value);

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      進捗率を設定
 *  \param[in]  identifier      実績の識別子
 *  \param[in]  progressRate    設定する進捗率
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool Server::SetProgressRate(Identifier identifier, float progressRate) noexcept
{
    if (auto *data = _delegate->GetAchievementData(identifier); data == nullptr)
    {
        return false;
    }

    _temporaryDataArray.emplace_back(identifier, progressRate);

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      達成状態を設定
 *  \param[in]  identifier      実績の識別子
 *  \param[in]  isAchieved      設定する達成状態
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool Server::SetAchieved(Identifier identifier, bool isAchieved) noexcept
{
    if (auto *data = _delegate->GetAchievementData(identifier); data == nullptr)
    {
        return false;
    }

    _temporaryDataArray.emplace_back(identifier, isAchieved);

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      実績の進捗率を取得
 *  \param[in]  identifier      実績の識別子
 *  \return     現在の進捗率
 */
/* ------------------------------------------------------------------------- */
float Server::GetProgressRate(Identifier identifier) const noexcept
{
    auto *data = _delegate->GetAchievementData(identifier);
    if (data == nullptr)
    {
        return 0;
    }

    return data->progressRate;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      実績の達成状況を取得
 *  \param[in]  identifier      実績の識別子
 *  \retval     true            達成済み
 *  \retval     false           未達成
 */
/* ------------------------------------------------------------------------- */
bool Server::IsAchieved(Identifier identifier) const noexcept
{
    auto *data = _delegate->GetAchievementData(identifier);
    if (data == nullptr)
    {
        return false;
    }

    return data->isAchieved;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      実績の達成状況を整数値で取得
 *  \param[in]  identifier      実績の識別子
 *  \return     現在の達成状況
 */
/* ------------------------------------------------------------------------- */
uint32_t Server::GetValue(Identifier identifier) const noexcept
{
    auto *data = _delegate->GetAchievementData(identifier);
    if (data == nullptr)
    {
        return 0;
    }

    return data->intValue;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      実績のクリア
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool Server::Clear() noexcept
{
    return _delegate->Clear();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フレーム更新処理
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       未使用
 */
/* ------------------------------------------------------------------------- */
void Server::OnPreFrameUpdate(void *callbackArg, [[maybe_unused]] void *notifyArg) noexcept
{
    auto *server = static_cast<Server *>(callbackArg);

    switch (server->_state)
    {
        // 初期化中
        case State::Initialize:
            if (server->_delegate->IsInitialized())
            {
                server->_delegate->FetchRequests();
                server->_state = State::Fetch;
            }
            break;

        // 待機
        case State::Idle:
            if (!server->_temporaryDataArray.empty())
            {
                bool hasUpdatedData = false;
                for (auto &data : server->_temporaryDataArray)
                {
                    if (server->ApplyTemporaryData(data))
                    {
                        hasUpdatedData = true;
                    }
                }
                server->_temporaryDataArray.clear();

                if (hasUpdatedData)
                {
                    server->_delegate->ReportRequests();
                    server->_state = State::Report;
                }
            }
            break;

        // データ取得中
        case State::Fetch:
            if (!server->_delegate->IsProcessing())
            {
                server->_state = State::Idle;
            }
            break;

        // データ送信中
        case State::Report:
            if (!server->_delegate->IsProcessing())
            {
                server->_state = State::Idle;
            }
            break;
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テンポラリデータを反映
 *  \param[in]  data        反映するテンポラリデータ
 *  \retval     true        値が更新された
 *  \retval     false       値が更新されなかった
 */
/* ------------------------------------------------------------------------- */
bool Server::ApplyTemporaryData(const TemporaryData &data) noexcept
{
    switch (data.type)
    {
        case ValueType::Boolean:
            return ApplyAchieved(data.identifier, data.isAchieved);

        case ValueType::Integer:
            return ApplyValue(data.identifier, data.intValue);

        case ValueType::ProgressRate:
            return ApplyProgressRate(data.identifier, data.progressRate);
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      値を反映
 *  \param[in]  identifier      実績の識別子
 *  \param[in]  value           反映する値
 *  \retval     true            値が更新された
 *  \retval     false           値が更新されなかった
 */
/* ------------------------------------------------------------------------- */
bool Server::ApplyValue(Identifier identifier, uint32_t value) noexcept
{
    // デリゲートからデータを取得
    auto *data = _delegate->GetAchievementData(identifier);
    if (data == nullptr)
    {
        return false;
    }

    // 達成済みなら何もしない
    if (data->isAchieved)
    {
        return false;
    }

    bool isUpdated = false;

    // 数値のタイプに応じて書き換える値を変更
    switch (data->type)
    {
        // 整数値を達成状態に変換して設定
        case ValueType::Boolean:
            if (value != 0)
            {
                data->isAchieved = true;
                isUpdated = true;
            }
            break;

        // 整数値をそのまま設定
        case ValueType::Integer:
            if (data->intValue < value)
            {
                data->intValue = value;
                isUpdated = true;
            }
            break;

        // 整数値を進捗率に変換して設定
        case ValueType::ProgressRate:
            if (auto progressRate = static_cast<float>(value); data->progressRate < progressRate)
            {
                data->progressRate = progressRate;
                isUpdated = true;
            }
            break;
    }

    // 値が書き換えられていたら実績データを更新する
    if (isUpdated)
    {
        UpdateAchievementData(data);
    }

    return isUpdated;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      進捗率を反映
 *  \param[in]  identifier      実績の識別子
 *  \param[in]  progressRate    反映する進捗率
 *  \retval     true            値が更新された
 *  \retval     false           値が更新されなかった
 */
/* ------------------------------------------------------------------------- */
bool Server::ApplyProgressRate(Identifier identifier, float progressRate) noexcept
{
    // デリゲートからデータを取得
    auto *data = _delegate->GetAchievementData(identifier);
    if (data == nullptr)
    {
        return false;
    }

    // 達成済みなら何もしない
    if (data->isAchieved)
    {
        return false;
    }

    bool isUpdated = false;

    // 数値のタイプに応じて書き換える値を変更
    switch (data->type)
    {
        // 進捗率を達成状態に変換して設定
        case ValueType::Boolean:
            data->isAchieved = progressRate >= 100.0f;
            isUpdated = data->isAchieved;
            break;

        // 進捗率を整数値に変換して設定
        case ValueType::Integer:
            if (auto value = static_cast<uint32_t>(static_cast<float>(data->max) * progressRate);
                data->intValue < value)
            {
                data->intValue = value;
                isUpdated = true;
            }
            break;

        // 進捗率をそのまま設定
        case ValueType::ProgressRate:
            if (data->progressRate < progressRate)
            {
                data->progressRate = progressRate;
                isUpdated = true;
            }
            break;
    }

    // 値が書き換えられていたら実績データを更新する
    if (isUpdated)
    {
        UpdateAchievementData(data);
    }

    return isUpdated;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      達成状態を反映
 *  \param[in]  identifier      実績の識別子
 *  \param[in]  isAchieved      達成状態
 *  \retval     true            値が更新された
 *  \retval     false           値が更新されなかった
 */
/* ------------------------------------------------------------------------- */
bool Server::ApplyAchieved(Identifier identifier, bool isAchieved) noexcept
{
    // デリゲートからデータを取得
    auto *data = _delegate->GetAchievementData(identifier);
    if (data == nullptr)
    {
        return false;
    }

    // 達成済みなら何もしない
    if (data->isAchieved)
    {
        return false;
    }

    bool isUpdated = false;

    // 数値のタイプに応じて書き換える値を変更
    switch (data->type)
    {
        // 達成状態をそのまま設定
        case ValueType::Boolean:
            data->isAchieved = isAchieved;
            isUpdated = isAchieved;
            break;

        // 達成状態を整数値に変換して設定
        case ValueType::Integer:
            if (isAchieved)
            {
                data->intValue = data->max;
                isUpdated = true;
            }
            break;

        // 達成状態を進捗率に変換して設定
        case ValueType::ProgressRate:
            if (isAchieved)
            {
                data->progressRate = 100.0f;
                isUpdated = true;
            }
            break;
    }

    // 値が書き換えられていたら実績データを更新する
    if (isUpdated)
    {
        UpdateAchievementData(data);
    }

    return isUpdated;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          実績データの更新
 *  \param[in,out]  data    更新する実績データ
 */
/* ------------------------------------------------------------------------- */
void Server::UpdateAchievementData(AchievementData *data) noexcept
{
    switch (data->type)
    {
        // Booleanタイプでの整数値と進捗率を設定
        case ValueType::Boolean:
            data->intValue = data->isAchieved ? 1 : 0;
            data->progressRate = data->isAchieved ? 100.0f : 0.0f;
            break;

        // 整数タイプでの達成率と達成状況を設定
        case ValueType::Integer:
            if (data->intValue >= data->max)
            {
                data->intValue = data->max;
                data->progressRate = 100.0f;
                data->isAchieved = true;
            }
            else
            {
                data->progressRate = static_cast<float>(data->intValue) / static_cast<float>(data->max);
                data->isAchieved = false;
            }
            break;

        // 進捗率タイプでの整数値と達成状況を設定
        case ValueType::ProgressRate:
            if (data->progressRate >= 100.0f)
            {
                data->progressRate = 100.0f;
                data->intValue = 100;
                data->isAchieved = true;
            }
            else
            {
                data->intValue = static_cast<uint32_t>(data->progressRate);
                data->isAchieved = false;
            }
            break;
    }

    // このデータが更新された事をマークする
    data->isUpdated = true;
}
}    // namespace MGL::Achievement

// vim: et ts=4 sw=4 sts=4
