// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_ui_widget.h
 *  \brief      MGL UIウィジット
 *  \date       Since: July 5, 2021. 7:59:33 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_UI_WIDGET_H_1625439573
#define INCGUARD_MGL_UI_WIDGET_H_1625439573

#include <mgl/common/mgl_alignment.h>
#include <mgl/input/touch/mgl_touch_state.h>
#include <mgl/mgl_environment.h>
#include <mgl/stl/mgl_stl_containers.h>
#include <mgl/ui/mgl_ui_event_delegate.h>
#include <mgl/ui/mgl_ui_event_listener.h>

namespace MGL::UI
{
class EventContext;
class EventListener;
class EventDelegate;

//! UIウィジットクラス
class Widget
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    Widget(uint32_t typeIdentifier) noexcept
        : _typeIdentifier(typeIdentifier)
    {
    }

    virtual ~Widget() noexcept = default;

    void Update(const Vector2 &offset = Vector2(0.0f, 0.0f), bool updateEvent = true) noexcept;
    void Render(const Vector2 &offset = Vector2(0.0f, 0.0f)) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      更新処理（オフセットなし）
     *  \param[in]  updateEvent イベント更新を行うかのフラグ
     */
    /* ------------------------------------------------------------------------- */
    void Update(bool updateEvent) noexcept
    {
        Update(Vector2(), updateEvent);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      矩形の取得
     *  \return     このウィジットの矩形
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual Rectangle GetRectangle() const noexcept
    {
        return {};
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィジットの種類の識別子を取得
     *  \return     識別子
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr uint32_t GetTypeIdentifier() const noexcept
    {
        return _typeIdentifier;
    }

    void SetVisible(bool isVisible) noexcept;
    [[nodiscard]] bool IsVisible() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このウィジットに設定された可視状態のフラグを取得
     *  \return     可視状態のフラグ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool GetLocalVisibleFlag() const noexcept
    {
        return _isVisible;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      透過値の設定
     *  \param[in]  transparency   設定する透過値
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetTransparency(float transparency) noexcept
    {
        _transparency = transparency;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このウィジットに設定された透過値の設定
     *  \return     このウィジットの透過値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr float GetLocalTransparency() const noexcept
    {
        return _transparency;
    }

    [[nodiscard]] float GetTransparency() const noexcept;

    [[nodiscard]] Vector2 GetPosition() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      サイズを取得
     *  \return     サイズ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] MGL_MAYBE_CONSTEXPR Vector2 GetSize() const noexcept
    {
        return GetRectangle().GetSize();
    }

    [[nodiscard]] Vector2 GetParentSize() const noexcept;

    bool AddChild(Widget *child) noexcept;
    bool RemoveChild(Widget *child) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      親ウィジットの取得
     *  \return     親ウィジット
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const Widget *GetParent() const noexcept
    {
        return _parent;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      子ウィジットの取得
     *  \return     子ウィジットのリスト
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const STL::list<Widget *> &GetChildren() const noexcept
    {
        return _children;
    }

    bool ActivateEventContext(EventListener *listener, EventDelegate *delegate) noexcept;
    bool ActivateEventContext(EventListener *listener) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      イベントIDを設定
     *  \param[in]  identifier  設定するイベントID
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetEventIdentifier(EventID identifier) noexcept
    {
        _eventIdentifier = identifier;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      イベントIDを取得
     *  \return     イベントID
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr EventID GetEventIdentifier() const noexcept
    {
        return _eventIdentifier;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィジットの中心アライメントを設定
     *  \param[in]  pivotAlignment  中心位置のアライメント
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetPivot(const Alignment &pivotAlignment) noexcept
    {
        _pivotAlignment = pivotAlignment;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィジットの中心アライメントを取得
     *  \return     中心位置のアライメント
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Alignment &GetPivot() const noexcept
    {
        return _pivotAlignment;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィジットのアンカーを設定
     *  \param[in]  anchorAlignment     アンカー設定
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetAnchor(const Alignment &anchorAlignment) noexcept
    {
        _anchorAlignment = anchorAlignment;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィジットのアンカー取得
     *  \return     アンカー設定
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Alignment &GetAnchor() const noexcept
    {
        return _anchorAlignment;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      イベントのステートを取得
     *  \return     イベントのステート
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr EventState GetEventState() const noexcept
    {
        return _eventState;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      イベントのタッチIDを取得
     *  \return     イベントのタッチID
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr Input::TouchID GetEventTouchID() const noexcept
    {
        return _eventTouchID;
    }

protected:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      更新処理
     *  \param[in]  position    このウィジットの位置
     */
    /* ------------------------------------------------------------------------- */
    virtual void OnUpdate(const Vector2 &position) noexcept
    {
        (void)position;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      描画処理
     *  \param[in]  position    このウィジットの位置
     */
    /* ------------------------------------------------------------------------- */
    virtual void OnRender(const Vector2 &position) noexcept
    {
        (void)position;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      イベント通知後の処理
     *  \param[in]  eventType   通知したイベントの種類
     *  \param[in]  argument    イベント引数
     */
    /* ------------------------------------------------------------------------- */
    virtual void OnNotifiedEvent(EventType eventType, uint32_t argument) noexcept
    {
        (void)eventType;
        (void)argument;
    }

private:
    void UpdateEvent(EventContext *eventContext, const Vector2 &offset) noexcept;
    void ApplyEventResult(EventContext *eventContext, const EventResult &result) noexcept;
    void SendEvent(EventContext *eventContext, EventType eventType, uint32_t argument = 0, Input::TouchID touchID = Input::TouchID::Invalid) noexcept;

    [[nodiscard]] Vector2 GetAdjustedPosition() const noexcept;

    uint32_t _typeIdentifier;

    Widget *_parent{nullptr};
    STL::list<Widget *> _children;

    bool _isVisible{true};
    float _transparency{1.0f};

    STL::unique_ptr<EventContext> _eventContext{nullptr};
    EventState _eventState{EventState::None};
    EventID _eventIdentifier{kInvalidEventID};
    Input::TouchID _eventTouchID{Input::TouchID::Invalid};

    Alignment _pivotAlignment;
    Alignment _anchorAlignment;
};

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      型情報付きのウィジットクラス
 *  \tparam     Type    識別子に関連付ける型
 */
/* ------------------------------------------------------------------------- */
template<typename Type>
class TypedWidget : public Widget
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  type    識別子
     */
    /* ------------------------------------------------------------------------- */
    TypedWidget(Type type) noexcept
        : Widget(static_cast<uint32_t>(type))
    {
    }

    ~TypedWidget() noexcept override = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      識別子を取得
     *  \return     識別子
     */
    /* ------------------------------------------------------------------------- */
    constexpr Type GetType() const noexcept
    {
        return static_cast<Type>(GetTypeIdentifier());
    }
};
}    // namespace MGL::UI

#endif    // INCGUARD_MGL_UI_WIDGET_H_1625439573

// vim: et ts=4 sw=4 sts=4
