// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_ui_input_scope.h
 *  \brief      MGL UI入力スコープ
 *  \date       Since: October 28, 2023. 22:19:39 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_UI_INPUT_SCOPE_H_1698499179
#define INCGUARD_MGL_UI_INPUT_SCOPE_H_1698499179

#include <mgl/math/mgl_rectangle.h>

namespace MGL::UI
{
//! 有効範囲外に出た際の挙動
enum class ScopeOutBehavior : uint8_t
{
    Unfocus,    //!< フォーカス状態を解除
    Keep        //!< フォーカス状態を保持
};

//! 入力の有効範囲
class InputScope
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr InputScope() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  scope       入力範囲
     *  \param[in]  behavior    入力範囲外に出た際の挙動
     */
    /* ------------------------------------------------------------------------- */
    constexpr InputScope(const Rectangle &scope,
                         ScopeOutBehavior behavior = ScopeOutBehavior::Unfocus) noexcept
        : _isValid(scope.width <= 0.0f || scope.height <= 0.0f)
        , _scope(scope)
        , _behavior(behavior)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      有効範囲を設定
     *  \param[in]  scope   有効範囲
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetScope(const Rectangle &scope) noexcept
    {
        _scope = scope;
        _isValid = (scope.width <= 0.0f || scope.height <= 0.0f);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      入力範囲外に出た際の挙動を設定
     *  \param[in]  behavior   挙動
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetBehavior(ScopeOutBehavior behavior) noexcept
    {
        _behavior = behavior;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      入力範囲外に出た際の挙動を取得
     *  \return     挙動
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr ScopeOutBehavior GetBehavior() const noexcept
    {
        return _behavior;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定した座標が入力範囲内に入っているかを取得
     *  \param[in]  position    座標
     *  \retval     true        入力範囲内
     *  \retval     false       入力範囲外
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsEnclosed(const Vector2 &position) const noexcept
    {
        return _scope.IsEnclosed(position);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      有効な入力範囲が設定されているかを取得
     *  \retval     true    有効
     *  \retval     false   向こう
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsValid() const noexcept
    {
        return _isValid;
    }

private:
    bool _isValid{false};
    Rectangle _scope;
    ScopeOutBehavior _behavior{ScopeOutBehavior::Unfocus};
};

}    // namespace MGL::UI

#endif    // INCGUARD_MGL_UI_INPUT_SCOPE_H_1698499179

// vim: et ts=4 sw=4 sts=4
