// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_ui_event_defs.h
 *  \brief      MGL UIイベント関連定義
 *  \date       Since: July 7, 2023. 11:06:27 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_UI_EVENT_DEFS_H_1688695587
#define INCGUARD_MGL_UI_EVENT_DEFS_H_1688695587

#include <mgl/common/mgl_bit.h>
#include <mgl/input/touch/mgl_touch_state.h>
#include <mgl/math/mgl_rectangle.h>

namespace MGL::UI
{
class Widget;

//! イベントタイプ
enum class EventType : uint8_t
{
    None,                //!< なし
    Focus,               //!< フォーカス
    Unfocus,             //!< フォーカス解除
    Decide,              //!< 決定
    DecideAndUnfocus,    //!< 決定後にフォーカス解除
};

//! イベントステート
enum class EventState : uint8_t
{
    None,        //!< 状態なし
    Focusing,    //!< フォーカス中
};

using EventID = int32_t;
inline constexpr const EventID kInvalidEventID = -1;

//! イベント対象の情報
struct EventTarget
{
    Widget *widget;         //!< 対象ウィジット
    Rectangle rectangle;    //!< 判定用の矩形(グローバル座標系)

    //! コンストラクタ
    constexpr EventTarget(Widget *widget_, const Rectangle &rectangle_) noexcept
        : widget(widget_)
        , rectangle(rectangle_)
    {
    }
};

//! イベント結果
struct EventResult
{
    EventType type;            //!< 発生イベントの種類
    EventID identifier;        //!< ウィジットのイベントID
    uint32_t argument;         //!< イベント引数
    Input::TouchID touchID;    //!< タッチID（タッチ入力の場合のみ使用）

    //! コンストラクタ
    constexpr EventResult() noexcept
        : type(EventType::None)
        , identifier(kInvalidEventID)
        , argument(0)
        , touchID(MGL::Input::TouchID::Invalid)
    {
    }

    //! コンストラクタ
    constexpr EventResult(EventType type_,
                          EventID identifier_,
                          uint32_t argument_ = 0,
                          Input::TouchID touchID_ = MGL::Input::TouchID::Invalid) noexcept
        : type(type_)
        , identifier(identifier_)
        , argument(argument_)
        , touchID(touchID_)
    {
    }
};

//! イベントによる移動先
struct EventMoveLink
{
    MGL::UI::EventID value{kInvalidEventID};    //!< リンク先のID
    bool isLoop{false};                         //!< ループ移動フラグ

    //! bool型へのキャスト
    explicit constexpr operator bool() const noexcept
    {
        return value != MGL::UI::kInvalidEventID;
    }
};

//! 上下左右の移動先テーブル
struct EventMoveTable
{
    EventMoveLink up;       //!< 上入力時のリンクID
    EventMoveLink down;     //!< 下入力時のリンクID
    EventMoveLink left;     //!< 左入力時のリンクID
    EventMoveLink right;    //!< 右入力時のリンクID
};

//! ユーザー入力
enum class UserInput : uint8_t
{
    MoveUp,       //!< 上移動
    MoveDown,     //!< 下移動
    MoveLeft,     //!< 左移動
    MoveRight,    //!< 右移動
    Decide,       //!< 決定
    Cancel,       //!< キャンセル

    MoveUpHasARepeat,       //!< 上移動がリピート入力かのフラグ
    MoveDownHasARepeat,     //!< 下移動がリピート入力かのフラグ
    MoveLeftHasARepeat,     //!< 左移動がリピート入力かのフラグ
    MoveRightHasARepeat,    //!< 右移動がリピート入力かのフラグ
};

//! ユーザー入力のビットフラグ
using UserInputFlags = EnumBitFlags<UserInput, uint32_t>;

}    // namespace MGL::UI
#endif    // INCGUARD_MGL_UI_EVENT_DEFS_H_1688695587

// vim: et ts=4 sw=4 sts=4
