// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_ui_event_default_delegate.h
 *  \brief      MGL UIイベントのデフォルトデリゲート
 *  \date       Since: July 14, 2023. 16:29:29 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_UI_EVENT_DEFAULT_DELEGATE_H_1689319769
#define INCGUARD_MGL_UI_EVENT_DEFAULT_DELEGATE_H_1689319769

#include <mgl/math/mgl_rectangle.h>
#include <mgl/stl/mgl_stl_containers.h>
#include <mgl/ui/mgl_ui_event_delegate.h>
#include <mgl/ui/mgl_ui_input_scope.h>

namespace MGL::UI
{

//! UIイベントのデフォルトデリゲート
class DefaultEventDelegate : public EventDelegate
{
public:
    ~DefaultEventDelegate() noexcept override = default;

    [[nodiscard]] virtual UserInputFlags OnUpdateUserInput() const noexcept;

    EventResult OnUpdateUIEvent(const STL::vector<EventTarget> &targets, EventID currentFocusID) noexcept final;
    static EventResult UpdateWithoutMove(const STL::vector<EventTarget> &targets, EventID currentFocusID, const InputScope &scope) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フォーカスの要求
     *  \param[in]  eventID     フォーカスするイベントID
     */
    /* ------------------------------------------------------------------------- */
    constexpr void FocusRequests(EventID eventID) noexcept
    {
        _focusRequestID = eventID;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      有効範囲の設定
     *  \param[in]  scope   有効範囲
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetScope(const InputScope &scope) noexcept
    {
        _scope = scope;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウスとタッチの有効フラグを設定
     *  \param[in]  isEnabled   有効フラグ
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetEnabledMouseAndTouch(bool isEnabled) noexcept
    {
        _isEnabledMouseAndTouch = isEnabled;
    }

    void SetMoveTable(EventID target, const EventMoveTable &table) noexcept;
    EventMoveTable &GetMoveTable(EventID target) noexcept;
    [[nodiscard]] bool HasMoveTable(EventID target) const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      初回のキー入力時のリンクテーブルを設定
     *  \param[in]  table   設定するリンクテーブル
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetFirstInputMoveTable(const EventMoveTable &table) noexcept
    {
        _firstInputMoveTable = table;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      初回のキー入力時のリンクテーブルを取得
     *  \return     リンクテーブル
     */
    /* ------------------------------------------------------------------------- */
    constexpr EventMoveTable &GetFirstInputMoveTable() noexcept
    {
        return _firstInputMoveTable;
    }

    void ClearTable() noexcept;

    static EventResult UpdateKeyInput(UserInputFlags input, const EventMoveTable &table) noexcept;

private:
    static EventResult UpdateMouseEvent(const STL::vector<EventTarget> &targets, EventID currentFocusID, const InputScope &scope) noexcept;
    static EventResult UpdateTouchEvent(const STL::vector<EventTarget> &targets, EventID currentFocusID, const InputScope &scope) noexcept;

    EventID _focusRequestID{kInvalidEventID};
    InputScope _scope;

    bool _isEnabledMouseAndTouch{false};

    STL::unordered_map<EventID, EventMoveTable> _moveTable;
    EventMoveTable _firstInputMoveTable;
};
}    // namespace MGL::UI

#endif    // INCGUARD_MGL_UI_EVENT_DEFAULT_DELEGATE_H_1689319769

// vim: et ts=4 sw=4 sts=4
