// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_text_stream.h
 *  \brief      テキストストリームクラス
 *  \date       Since: October 4, 2018. 2:13:19 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TEXT_STREAM_H_1538586799
#define INCGUARD_MGL_TEXT_STREAM_H_1538586799

#include <mgl/file/mgl_file_path_view.h>
#include <mgl/stl/mgl_stl_memory.h>
#include <mgl/text/mgl_text_character.h>
#include <mgl/text/mgl_text_defs.h>

namespace MGL::Text
{
//! テキストストリームクラス
class TextStream
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr TextStream() noexcept = default;

    TextStream(const uint8_t *buffer, size_t size, Encoding encoding = Encoding::Auto, bool existBOM = false) noexcept;

    void Set(const uint8_t *buffer, size_t size, Encoding encoding = Encoding::Auto, bool existBOM = false) noexcept;
    bool Load(const File::PathView &filepath, Encoding encoding = Encoding::Auto) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エンコードタイプの取得
     *  \return     設定されているエンコードタイプ
     *  \note
     *      この関数で取得できるエンコードタイプは，引数で指定または自動認識によって判別された値であり，
     *      関連づけられたバッファのエンコードタイプを正確に表しているものではない．
     *      両者が一致していない場合，Read()またはReadLine()に失敗する．
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] Encoding GetEncoding() const noexcept
    {
        return _encoding;
    }

    Character Read() noexcept;

    int Read(CharacterArray &array) noexcept;
    int ReadLine(CharacterArray &array) noexcept;

    void PrintInfo() const noexcept;


    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      終了判定
     *  \retval     true    終了
     *  \retval     false   バッファに残りがある
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsTerminated() const noexcept
    {
        return (_offset >= _size);
    }

private:
    Encoding CheckEncoding(bool &existBOM) const noexcept;
    [[nodiscard]] size_t GetStartOffset() const noexcept;

    Character ReadUTF8() noexcept;
    Character ReadUTF16() noexcept;
    Character ReadUTF32() noexcept;

    const uint8_t *_buffer{nullptr};
    size_t _size{0};
    Encoding _encoding{Encoding::Auto};
    size_t _offset{0};
    bool _existBOM{false};
    STL::unique_ptr<uint8_t[]> _loadBuffer{nullptr};
};
}    // namespace MGL::Text


#endif    // INCGUARD_MGL_TEXT_STREAM_H_1538586799

// vim: et ts=4 sw=4 sts=4
