// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_text_index_converter.h
 *  \brief      MGL テキストのインデックス化コンバータ
 *  \date       Since: June 27, 2021. 5:31:10 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TEXT_INDEX_CONVERTER_H_1624739470
#define INCGUARD_MGL_TEXT_INDEX_CONVERTER_H_1624739470

#include <mgl/render/font/mgl_font_defs.h>
#include <mgl/stl/mgl_stl_containers.h>
#include <mgl/stl/mgl_stl_memory.h>
#include <mgl/stl/mgl_stl_string.h>
#include <mgl/text/mgl_text_defs.h>

namespace MGL::Text
{
//! インデックスコンバータクラス
class IndexConverter
{
public:
    virtual ~IndexConverter() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      文字をインデックス文字に変換
     *  \param[in]  character   変換元の文字（UTF-32）
     *  \param[in]  faceType    フェイスタイプ
     *  \return     インデックス文字
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual Text::IndexedCharacter ToIndexedCharacter(char32_t character, Render::FontFaceType faceType) const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      インデックス文字をUTF-32の文字に変換
     *  \param[in]  indexedCharacter   インデックス文字
     *  \param[in]  faceType           フェイスタイプ
     *  \return     インデックスに対応したUTF-32エンコードの文字
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual char32_t ToUTF32(Text::IndexedCharacter indexedCharacter, Render::FontFaceType faceType) const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したフェイスを保持しているかを取得
     *  \param[in]  faceType    フェイスタイプ
     *  \retval     true        保持している
     *  \retval     false       保持していない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool HasFontFace(Render::FontFaceType faceType) const noexcept = 0;

    STL::unique_ptr<Text::IndexedCharacter[]> ToIndexedString(const char *text, bool enableFormat, bool enableTag) const noexcept;

private:
    const char *ParseSubstitute(STL::vector<Text::IndexedCharacter> &indexes, bool enable, const char *text, Render::FontFaceType faceType) const noexcept;
    const char *ParseTag(STL::vector<Text::IndexedCharacter> &indexes, Render::FontFaceType &faceType, bool enable, const char *text) const noexcept;
    static void ParseColorTag(STL::vector<Text::IndexedCharacter> &indexes, const STL::string &tagName) noexcept;
    void ParseFaceTag(STL::vector<Text::IndexedCharacter> &indexes, Render::FontFaceType &faceType, const STL::string &tagName) const noexcept;
    bool ParseRuby(STL::vector<Text::IndexedCharacter> &indexes, Render::FontFaceType faceType, const STL::string &tagName) const noexcept;
    static void ParseResetTag(STL::vector<Text::IndexedCharacter> &indexes, Render::FontFaceType &faceType, const STL::string &tagName) noexcept;
    static const char *GetTagName(STL::string &tagName, const char *text) noexcept;
    static const char *GetUTF32(char32_t &u32char, const char *text) noexcept;
};
}    // namespace MGL::Text

#endif    // INCGUARD_MGL_TEXT_INDEX_CONVERTER_H_1624739470

// vim: et ts=4 sw=4 sts=4
