// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_text_format_argument.h
 *  \brief      MGL テキストフォーマットの引数
 *  \date       Since: May 28, 2021. 22:25:20 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TEXT_FORMAT_ARGUMENT_H_1622208320
#define INCGUARD_MGL_TEXT_FORMAT_ARGUMENT_H_1622208320

#include <type_traits>
#include <utility>

#include <mgl/file/mgl_file_path_view.h>
#include <mgl/stl/mgl_stl_containers.h>
#include <mgl/stl/mgl_stl_string.h>
#include <mgl/text/mgl_text_defs.h>
#include <mgl/text/mgl_text_format_options.h>

namespace MGL::Text
{
//! テキストフォーマットの引数
class FormatArgument
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    FormatArgument() noexcept
        : _type(Type::Void)
        , _i(0)
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \tparam     T       引数の型
     *  \param[in]  value   引数の値
     */
    /* ------------------------------------------------------------------------- */
    template <typename T>
    FormatArgument(const T &value) noexcept
        : _type(Type::Void)
        , _i(0)
    {
        // 整数
        if constexpr (std::is_integral_v<T>)
        {
            // 符号付き
            if constexpr (std::is_signed_v<T>)
            {
                _type = Type::Int;
                _i = value;
            }
            // 符号なし
            else
            {
                _type = Type::UnsignedInt;
                _ui = value;
            }
        }
        // 浮動小数点数
        else if constexpr (std::is_floating_point_v<T>)
        {
            _type = Type::Float;
            _f = value;
        }
        // bool型
        else if constexpr (std::is_same_v<T, bool>)
        {
            _type = Type::Bool;
            _b = value;
        }
        // MGL::STL::string型
        else if constexpr (std::is_same_v<T, STL::string>)
        {
            _type = Type::String;
            _i = 0;
            _string = value;
        }
        // その他 const char *型に変換可能な型
        else if constexpr (std::is_convertible_v<T, const char *>)
        {
            _type = Type::String;
            _i = 0;
            _string = value;
        }
        // 上記以外のポインタ型
        else if constexpr (std::is_pointer_v<T>)
        {
            _type = Type::Pointer;
            _ui = reinterpret_cast<uint64_t>(value);
        }
        // それ以外は不明
        else
        {
            static_assert(true, "Unknown format type detected. Please cast int, float, or const char * type.");
        }
    }

    [[nodiscard]] STL::string ToString(const FormatOptions &options = FormatOptions()) const noexcept;

    static const char *Parse(FormatOptions &options, const char *parseText) noexcept;

    static STL::vector<IndexedCharacter> ToIndexedCharacters(const FormatOptions &options) noexcept;

private:
    //! 値の型
    enum class Type : uint8_t
    {
        Void,           //!< 型なし
        Int,            //!< 符号付き整数
        UnsignedInt,    //!< 符号なし整数
        Float,          //!< 浮動小数点数
        Bool,           //!< 論理値
        String,         //!< 文字列
        Pointer,        //!< ポインタ
    };

    [[nodiscard]] bool IsZero() const noexcept;
    [[nodiscard]] bool IsPositiveValue() const noexcept;
    [[nodiscard]] bool IsNegativeValue() const noexcept;
    [[nodiscard]] STL::string ToStringFromNumber(const FormatOptions &options) const noexcept;
    [[nodiscard]] STL::string ToStringFromDecimal(const FormatOptions &options) const noexcept;
    [[nodiscard]] STL::string ToHexStringFromValue(const FormatOptions &options) const noexcept;

    static void AlignmentString(STL::string &result, const FormatOptions &options, const STL::string &string) noexcept;

    Type _type;
    union
    {
        int64_t _i;
        uint64_t _ui;
        float _f;
        bool _b;
    };

    STL::string _string;
};

//! テキストフォーマットの引数の配列
using FormatArgs = STL::vector<FormatArgument>;

}    // namespace MGL::Text

#endif    // INCGUARD_MGL_TEXT_FORMAT_ARGUMENT_H_1622208320

// vim: et ts=4 sw=4 sts=4
