// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_text_defs.h
 *  \brief      MGL テキスト関連各種宣言
 *  \date       Since: June 24, 2021. 16:36:54 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TEXT_DEFS_H_1624520214
#define INCGUARD_MGL_TEXT_DEFS_H_1624520214

#include <cstdint>

namespace MGL::Text
{
//! テキストのエンコードタイプ
enum class Encoding : uint8_t
{
    Auto,       //!< 自動認識
    UTF8,       //!< UTF-8
    UTF16BE,    //!< UTF-16 ビッグエンディアン
    UTF16LE,    //!< UTF-16 リトルエンディアン
    UTF32BE,    //!< UTF-32 ビッグエンディアン
    UTF32LE,    //!< UTF-32 リトルエンディアン
};

//! インデックス化文字
using IndexedCharacter = uint16_t;

constexpr IndexedCharacter kIndexedCharacterReserveStart = 0xD000;                    //!< 予約開始インデックス
constexpr IndexedCharacter kIndexedCharacterTagParameterStart = 0xD000;               //!< タグパラメータの開始インデックス
constexpr IndexedCharacter kIndexedCharacterTagParameterEnd = 0xDFFF;                 //!< タグパラメータの終了インデックス
constexpr IndexedCharacter kIndexedCharacterSubstituteOptionReserveStart = 0xE000;    //!< 置換フォーマットオプションの開始インデックス
constexpr IndexedCharacter kIndexedCharacterSubstituteOptionReserveEnd = 0xEFFF;      //!< 置換フォーマットオプションの終了インデックス
constexpr IndexedCharacter kIndexedCharacterColorReset = 0xFFD0;                      //!< 色のリセット
constexpr IndexedCharacter kIndexedCharacterFaceReset = 0xFFD1;                       //!< 書体をリセット
constexpr IndexedCharacter kIndexedCharacterTagReset = 0xFFDF;                        //!< 全てのタグをリセット
constexpr IndexedCharacter kIndexedCharacterNewLine = 0xFFF0;                         //!< 改行
constexpr IndexedCharacter kIndexedCharacterSubstitute = 0xFFF1;                      //!< 置換
constexpr IndexedCharacter kIndexedCharacterRubyStart = 0xFFF2;                       //!< ルビの開始
constexpr IndexedCharacter kIndexedCharacterWideRubyStart = 0xFFF3;                   //!< 幅広ルビの開始
constexpr IndexedCharacter kIndexedCharacterRubyDelimiter = 0xFFF4;                   //!< ルビの区切り文字
constexpr IndexedCharacter kIndexedCharacterRubyEnd = 0xFFF5;                         //!< ルビの終了
constexpr IndexedCharacter kIndexedCharacterIgnore = 0xFFFD;                          //!< 無視すべき文字
constexpr IndexedCharacter kIndexedCharacterInvalid = 0xFFFE;                         //!< 無効なインデックス
constexpr IndexedCharacter kIndexedCharacterEndOfText = 0xFFFF;                       //!< テキスト終了

// 置換フォーマットのマスク値
constexpr IndexedCharacter kSubstituteFormatMaskIndex = 0xE000;            //!< 置換フォーマットのインデックスのマスク値
constexpr IndexedCharacter kSubstituteFormatMaskWidth = 0xE100;            //!< 置換フォーマットの桁幅のマスク値
constexpr IndexedCharacter kSubstituteFormatMaskPrecision = 0xE200;        //!< 置換フォーマットの小数点以下の桁幅のマスク値
constexpr IndexedCharacter kSubstituteFormatMaskFillCharacter = 0xE300;    //!< 置換フォーマットの桁幅埋め文字のマスク値
constexpr IndexedCharacter kSubstituteFormatMaskHexWidth = 0xE400;         //!< 置換フォーマットの16進数の桁幅のマスク値
constexpr IndexedCharacter kSubstituteFormatMaskFlags = 0xE500;            //!< 置換フォーマットのフラグのマスク値

// 置換フォーマットのフラグオプションのビット構成
constexpr uint8_t kSubstituteFlagBitShowPlusSign = 0;          //!< kSubstituteFormatMaskFlags内の符号表示フラグのビット位置
constexpr uint8_t kSubstituteFlagBitShowSeparator = 1;         //!< kSubstituteFormatMaskFlags内の区切り文字表示フラグのビット位置
constexpr uint8_t kSubstituteFlagBitSkipAutoIndex = 2;         //!< kSubstituteFormatMaskFlags内の自動インデックススキップフラグのビット位置
constexpr uint8_t kSubstituteFlagBitValueDisplayModeLo = 3;    //!< kSubstituteFormatMaskFlags内の値の表示モードのビット位置（2ビットのうちの下位）
constexpr uint8_t kSubstituteFlagBitValueDisplayModeHi = 4;    //!< kSubstituteFormatMaskFlags内の値の表示モードのビット位置（2ビットのうちの上位）

// パラメータ付きタグのマスク値
constexpr IndexedCharacter kTagParameterColor = 0xD000;    //!< 文字色
constexpr IndexedCharacter kTagParameterFace = 0xD100;     //!< 書体

// 書体のインデックス値（0xD1xxの下位8ビットの値）
enum class FaceTypeIndex : uint8_t
{
    Default,    //!< デフォルト
    Ruby,       //!< ルビ
    Bold,       //!< ボールド
};
}    // namespace MGL::Text

#endif    // INCGUARD_MGL_TEXT_DEFS_H_1624520214

// vim: et ts=4 sw=4 sts=4
