// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_text_character.h
 *  \brief      文字クラス
 *  \date       Since: October 4, 2018. 2:19:01 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TEXT_CHARACTER_H_1538587141
#define INCGUARD_MGL_TEXT_CHARACTER_H_1538587141

#include <cstddef>
#include <cstdint>

#include <mgl/stl/mgl_stl_containers.h>
#include <mgl/stl/mgl_stl_string.h>

namespace MGL::Text
{
//! 文字クラス
class Character
{
public:
    //! 不正な文字を検出した際に書き込むビット位置
    static constexpr uint32_t kInvalidMarkerBit = 1UL << 31UL;

    //! コードの有効範囲
    static constexpr uint32_t kValidateMask = 0xFFFFFFFF & ~kInvalidMarkerBit;

    //! バイト配列のサイズ
    static constexpr size_t kByteArraySize = 5;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  code    文字の値
     */
    /* ------------------------------------------------------------------------- */
    constexpr Character(uint32_t code) noexcept
        : _code(code)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      値を取得
     *  \return     値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr char32_t GetCode() const noexcept
    {
        return static_cast<char32_t>(_code & kValidateMask);
    }


    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      値を設定
     *  \param[in]  code    値
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetCode(char32_t code) noexcept
    {
        _code = static_cast<uint32_t>(code);

        if (_code > 0x10FFFF)
        {
            SetInvalidMarker();
        }
    }


    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      不正なコードのマーカーを書き込む
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetInvalidMarker() noexcept
    {
        _code |= kInvalidMarkerBit;
    }


    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コードが正常かどうかを取得する
     *  \retval     true    正常
     *  \retval     false   不正
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsValid() const noexcept
    {
        return ((_code & kInvalidMarkerBit) == 0);
    }


    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      Null文字かどうかを判定
     *  \retval     true    Null文字
     *  \retval     false   Null文字ではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsNull() const noexcept
    {
        return (GetCode() == 0);
    }


    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      改行文字かどうかを取得
     *  \retval     true    改行文字
     *  \retval     false   改行文字ではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsNewLine() const noexcept
    {
        return ((_code == 0x0D) || (_code == 0x0A));
    }


    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スペースかどうかを取得
     *  \retval     true    スペース文字
     *  \retval     false   スペース文字ではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsSpace() const noexcept
    {
        return ((_code == ' ') || (_code == '\t'));
    }


    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      数値かどうかを取得
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsDigit() const noexcept
    {
        return ((_code >= '0') && (_code <= '9'));
    }

    [[nodiscard]] size_t GetByteSize() const noexcept;
    bool ToByte(char byteArray[kByteArraySize]) const noexcept;
    char *ToByte(char *dest, size_t destSize) const noexcept;
    [[nodiscard]] STL::string ToString() const noexcept;

private:
    uint32_t _code;
};


//! 文字の集合体クラス
class CharacterArray : public STL::vector<Character>
{
public:
    [[nodiscard]] STL::string ToString() const;
    [[nodiscard]] STL::string ToString(uint32_t start, uint32_t end) const;

    [[nodiscard]] int Find(uint32_t findCode, uint32_t start = 0) const;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      STL::vector::atのオーバーライド
     *  \param[in]  n   要素のインデックス
     *  \return     nに対応した要素
     *
     *  範囲外アクセスを行った場合は終了コードが返る．
     */
    /* ------------------------------------------------------------------------- */
    reference at(size_type n)
    {
        return (n >= size()) ? _endCode : STL::vector<Character>::at(n);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      STL::vector::atのオーバーライド（const版）
     *  \param[in]  n   要素のインデックス
     *  \return     nに対応した要素
     *
     *  範囲外アクセスを行った場合は終了コードが返る．
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const_reference at(size_type n) const
    {
        return (n >= size()) ? _endCode : STL::vector<Character>::at(n);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      配列アクセス
     *  \param[in]  pos   要素のインデックス
     *  \return     posに対応した要素
     *
     *  範囲外アクセスを行った場合は終了コードが返る．
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] reference operator[](size_type pos) noexcept
    {
        return CharacterArray::at(pos);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      配列アクセス（const版）
     *  \param[in]  pos   要素のインデックス
     *  \return     posに対応した要素
     *
     *  範囲外アクセスを行った場合は終了コードが返る．
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const_reference operator[](size_type pos) const noexcept
    {
        return CharacterArray::at(pos);
    }

private:
    inline static Character _endCode{'\0'};
};
}    // namespace MGL::Text

#endif    // INCGUARD_MGL_TEXT_CHARACTER_H_1538587141

// vim: et ts=4 sw=4 sts=4

