// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_task_weak_node.h
 *  \brief      MGL 弱参照タスクノード
 *  \date       Since: June 3, 2021. 10:48:12 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TASK_WEAK_NODE_H_1622684892
#define INCGUARD_MGL_TASK_WEAK_NODE_H_1622684892

#include <mgl/task/mgl_task_node.h>

namespace MGL::Task
{
//! 弱参照タスクノード
class WeakNode
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr WeakNode() noexcept = default;

    WeakNode(size_t listIndex) noexcept;
    WeakNode(const Node *node) noexcept;

    bool Set(size_t listIndex) noexcept;
    bool Set(const Node *node) noexcept;
    [[nodiscard]] Node *Get() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      参照先の有効状態を取得
     *  \retval     true    無効
     *  \retval     false   有効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsValid() const noexcept
    {
        return Get() != nullptr;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      有効状態をbool型へのキャストで取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    explicit operator bool() const noexcept
    {
        return IsValid();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      有効状態を否定演算子で取得
     *  \retval     true    無効
     *  \retval     false   有効
     */
    /* ------------------------------------------------------------------------- */
    bool operator!() const noexcept
    {
        return !IsValid();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したタスクノードにキャストして取得
     *  \tparam     T           キャスト先のクラス
     *  \tparam     identifier  Tと関連付けられたタスクノードID
     *  \return     参照先のIDと引数のIDが一致していればキャストしたタクスノードを返し，そうで無ければnullptrを返す
     *  \note
     *      キャスト先のクラスに静的なメンバ変数kTaskIdentifierが定義されている場合，
     *      テンプレート引数のidentifierを省略することでそちらを参照する．
     */
    /* ------------------------------------------------------------------------- */
    template <class T, Identifier identifier = Identifier(T::kTaskIdentifier)>
    [[nodiscard]] T *Get() const noexcept
    {
        if (auto *node = Get(); node != nullptr)
        {
            if (node->GetIdentifier() == identifier)
            {
                return reinterpret_cast<T *>(node);
            }
        }

        return nullptr;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タスクへのイベント通知
     *  \tparam     EventIDType イベントの種類を表す型
     *  \param[in]  event       通知するイベントの種類
     *  \param[in]  argument    通知先のタスクに渡す引数
     *  \retval     true        成功
     *  \retval     false       失敗
     */
    /* ------------------------------------------------------------------------- */
    template <typename EventIDType>
    bool NotifyEvent(EventIDType event, void *argument) noexcept
    {
        if (auto *node = Get(); node != nullptr)
        {
            node->NotifyEvent(event, argument);
            return true;
        }

        return false;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      参照先タスクを削除
     *  \param[in]  resideLevel 常駐レベル
     *  \retval     true        成功
     *  \retval     false       失敗
     */
    /* ------------------------------------------------------------------------- */
    bool Kill(ResideLevel resideLevel = ResideLevel::NoResident) noexcept
    {
        if (auto *node = Get(); node != nullptr)
        {
            if (node->GetResideLevel() <= resideLevel)
            {
                node->Kill();
                return true;
            }
        }

        return false;
    }

private:
    size_t _listIndex{0};
    Identifier _identifier{0};
    UniqueIdentifier _uniqueID{UniqueIdentifier::Invalid};
};

};    // namespace MGL::Task

#endif    // INCGUARD_MGL_TASK_WEAK_NODE_H_1622684892

// vim: et ts=4 sw=4 sts=4
