// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_task_thread_pool.h
 *  \brief      MGL タスクシステム用スレッドプール
 *  \date       Since: November 3, 2023. 22:14:21 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TASK_THREAD_POOL_H_1699017261
#define INCGUARD_MGL_TASK_THREAD_POOL_H_1699017261

#include <cstddef>

#include <mgl/stl/mgl_stl_containers.h>
#include <mgl/stl/mgl_stl_memory.h>
#include <mgl/task/mgl_task_thread.h>

namespace MGL::Task
{
class ThreadPool : public ThreadListener
{
public:
    int32_t Initialize(int32_t threadCount) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スレッドプールが利用可能かを取得
     *  \retval     true    利用可能
     *  \retval     false   利用不可
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsAvailable() const noexcept
    {
        return _isAvailable;
    }

    bool Execute(Node *node, ExecuteStage stage) noexcept;

    void WaitForExecutable() noexcept;
    void WaitForComplete() noexcept;
    void SyncBarrier(Identifier identifier) noexcept;

    void OnCompleted(uint32_t index, const Node *node) noexcept override;

private:
    struct Element
    {
        STL::unique_ptr<Thread> thread;
        int32_t next{-1};
    };

    bool _isAvailable{false};

    Element _freeTop;

    STL::vector<Element> _threads;
    uint32_t _executeCount{0};

    STL::unordered_map<Identifier, uint32_t> _barrierMap;

    std::mutex _mutex;
    std::mutex _conditionMutex;
    std::condition_variable _condition;
};
}    // namespace MGL::Task

#endif    // INCGUARD_MGL_TASK_THREAD_POOL_H_1699017261

// vim: et ts=4 sw=4 sts=4
