// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_task_thread.h
 *  \brief      MGL タスクシステム用スレッド
 *  \date       Since: November 3, 2023. 22:54:27 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TASK_THREAD_H_1699019667
#define INCGUARD_MGL_TASK_THREAD_H_1699019667

#include <atomic>
#include <condition_variable>
#include <mutex>
#include <thread>

#include <mgl/task/mgl_task_defs.h>

namespace MGL::Task
{
class Node;
class Thread;

//! スレッドからの通知を受けるためのリスナークラス
class ThreadListener
{
public:
    virtual ~ThreadListener() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      完了通知
     *  \param[in]  index   完了したスレッドのインデックス
     *  \param[in]  node    完了したタスクノード
     */
    /* ------------------------------------------------------------------------- */
    virtual void OnCompleted(uint32_t index, const Node *node) noexcept = 0;
};


//! タスクノード並列実行用スレッド
class Thread
{
public:
    Thread(ThreadListener &listener, uint32_t index) noexcept;

    ~Thread() noexcept;

    bool Create() noexcept;

    bool Execute(Node *taskNode, ExecuteStage stage) noexcept;

private:
    void Process() noexcept;

    ThreadListener &_listener;
    uint32_t _index;

    std::atomic<bool> _shouldExit{false};
    Node *_taskNode{nullptr};
    ExecuteStage _stage{0};

    std::atomic<bool> _isExecuting{false};

    std::mutex _mutex;
    std::condition_variable _condition;

    std::thread _thread;
};
}    // namespace MGL::Task

#endif    // INCGUARD_MGL_TASK_THREAD_H_1699019667

// vim: et ts=4 sw=4 sts=4
