/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_task_node_sublist.h
 *  \brief      MGL タスクリストのサブリスト
 *  \date       Since: March 30, 2023. 17:35:54 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TASK_NODE_SUBLIST_H_1680165354
#define INCGUARD_MGL_TASK_NODE_SUBLIST_H_1680165354

#include <condition_variable>
#include <mutex>

#include <mgl/task/mgl_task_node_list_element.h>
#include <mgl/task/mgl_task_thread_pool.h>
#include <mgl/task/mgl_task_weak_node.h>

namespace MGL::Task
{
//! タスクリストのサブリスト
class NodeSubList
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  identifier  タスクID
     */
    /* ------------------------------------------------------------------------- */
    NodeSubList(Identifier identifier) noexcept
        : _identifier(identifier)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タスクIDの取得
     *  \return     タスクID
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr Identifier GetIdentifier() const noexcept
    {
        return _identifier;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タスクの要素数を取得
     *  \return     タスクの要素数
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr size_t GetCount() const noexcept
    {
        return _count;
    }

    bool AddNodeListElement(NodeListElement *element) noexcept;
    void ActivateStandbyTask();
    void ExecuteAndRemove(ExecuteStage stage, NodeListElement *freeTop) noexcept;
    void ParallelExecute(ExecuteStage stage, ThreadPool &threadPool) noexcept;
    uint32_t Kill(ResideLevel resideLevel) noexcept;
    const NodeListElement *GetElement(const Node *node) noexcept;
    void GetNodeArray(STL::vector<WeakNode> &weakNodeArray) noexcept;
    void NotifyEvent(EventIdentifier eventIdentifier, void *argument) noexcept;

    void Remove(NodeListElement *freeTop) noexcept;

private:
    void Remove(NodeListElement *element, NodeListElement *freeTop) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      登録タスクを走査するためのテンプレート関数
     *  \note
     *      指定された関数functionを，各々のNodeListElement &を引数に渡して呼び出す．
     *      functionの戻り値がtrueであれば走査を継続し，falseなら中断する．
     */
    /* ------------------------------------------------------------------------- */
    template <typename Function>
    void ForEach(Function function) noexcept
    {
        for (auto *list : {_activeTop, _standbyTop})
        {
            if (list != nullptr)
            {
                std::unique_lock<std::mutex> lock(_mutex, std::defer_lock);
                if (list == _standbyTop)
                {
                    lock.lock();
                }

                auto *current = list;
                while (current != nullptr)
                {
                    if (!function(*current))
                    {
                        return;
                    }
                    current = current->next;
                }
            }
        }
    }

    Identifier _identifier;
    NodeListElement *_activeTop{nullptr};
    NodeListElement *_activeTail{nullptr};
    NodeListElement *_standbyTop{nullptr};
    NodeListElement *_standbyTail{nullptr};
    uint32_t _count{0};

    std::mutex _mutex;
};
}    // namespace MGL::Task

#endif    // INCGUARD_MGL_TASK_NODE_SUBLIST_H_1680165354

// vim: et ts=4 sw=4 sts=4
