// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_task_node_list.h
 *  \brief      MGL タスクノードのリスト
 *  \date       Since: June 1, 2021. 17:06:07 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TASK_NODE_LIST_H_1622534767
#define INCGUARD_MGL_TASK_NODE_LIST_H_1622534767

#include <mgl/common/mgl_singleton.h>
#include <mgl/stl/mgl_stl_containers.h>
#include <mgl/stl/mgl_stl_memory.h>
#include <mgl/task/mgl_task_node_sublist.h>
#include <mgl/task/mgl_task_thread_pool.h>

namespace MGL::Task
{
//! タスクノードのリスト
class NodeList final : public SharedSingleton<NodeList>
{
public:
    static STL::unique_ptr<NodeList> &GetInstanceRef() noexcept;

    bool Initialize(size_t capacity, const InitializeDescriptor &descriptor) noexcept;

    size_t Add(STL::unique_ptr<Node> addNode) noexcept;

    void Execute() noexcept;

    size_t Kill(ResideLevel resideLevel = ResideLevel::NoResident) noexcept;
    size_t Kill(Identifier identifier, ResideLevel resideLevel = ResideLevel::NoResident) noexcept;

    [[nodiscard]] size_t GetCount() const noexcept;
    [[nodiscard]] size_t GetCount(Identifier identifier) const noexcept;

    [[nodiscard]] const NodeListElement *GetElement(size_t index) const noexcept;
    const NodeListElement *GetElement(const Node *node) noexcept;

    void GetNodeArray(STL::vector<WeakNode> &weakNodeArray, Identifier identifier) noexcept;

    void NotifyEvent(EventIdentifier eventIdentifier, void *argument) noexcept;
    void NotifyEvent(Identifier identifier, EventIdentifier eventIdentifier, void *argument) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      追加可能なノードの数を取得
     *  \return     追加可能なノードの数
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr size_t GetCapacity() const noexcept
    {
        return _capacity;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      空きノードがあるかをチェック
     *  \retval     true    空きノードがある
     *  \retval     false   空きノードがない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool HasFreeNode() const noexcept
    {
        return _freeTopElement.next != nullptr;
    }

private:
    [[nodiscard]] NodeSubList *GetSubList(Identifier identifier) const noexcept;
    NodeSubList *CreateSubList(Identifier identifier) noexcept;

    size_t _capacity{0};
    STL::unique_ptr<NodeListElement[]> _nodeElements;
    NodeListElement _freeTopElement;
    std::mutex _mutex;

    STL::list<STL::unique_ptr<NodeSubList>> _subLists;

    UniqueIdentifier _currentUniqueId{UniqueIdentifier::Start};

    StageSettingsArray _stageSettingsArray;

    ThreadPool _threadPool;
};
}    // namespace MGL::Task

#endif    // INCGUARD_MGL_TASK_NODE_LIST_H_1622534767

// vim: et ts=4 sw=4 sts=4
