// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_task_node.h
 *  \brief      MGL タスクノード
 *  \date       Since: June 1, 2021. 16:46:33 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TASK_NODE_H_1622533593
#define INCGUARD_MGL_TASK_NODE_H_1622533593

#include <mgl/task/mgl_task_defs.h>

namespace MGL::Task
{
//! タスクノード
class Node
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  identifier  タスクノードのID
     */
    /* ------------------------------------------------------------------------- */
    constexpr Node(Identifier identifier) noexcept
        : _identifier(identifier)
    {
    }

    virtual ~Node() = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このタスクノードのIDを取得
     *  \return     タスクノードのID
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr Identifier GetIdentifier() const noexcept
    {
        return _identifier;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      削除要求
     */
    /* ------------------------------------------------------------------------- */
    constexpr void Kill() noexcept
    {
        _isRequestedKill = true;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      削除要求されているかを取得
     *  \retval     true    されている
     *  \retval     false   されていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsRequestedKill() const noexcept
    {
        return _isRequestedKill;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      常駐レベルの設定
     *  \param[in]  resideLevel     設定する常駐レベル
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetResideLevel(ResideLevel resideLevel) noexcept
    {
        _resideLevel = resideLevel;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      常駐レベルの取得
     *  \return     現在の常駐レベル
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr ResideLevel GetResideLevel() const noexcept
    {
        return _resideLevel;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      常駐タスクかどうかを取得
     *  \retval     true    常駐タスクである
     *  \retval     false   常駐タスクでない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsResident() const noexcept
    {
        return _resideLevel > ResideLevel::NoResident;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      非同期実行の設定
     *  \tparam     TaskIDType          タスクの種類を表す型
     *  \param[in]  isEnabled           有効フラグ
     *  \param[in]  barrierIdentifier   バリアID
     */
    /* ------------------------------------------------------------------------- */
    template <typename TaskIDType>
    constexpr void SetAsynchronous(bool isEnabled, TaskIDType barrierIdentifier = TaskIDType()) noexcept
    {
        _isAsynchronous = isEnabled;
        _barrierIdentifier = Identifier(barrierIdentifier);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      非同期実行が有効化されているかを取得
     *  \retval     true    有効化されている
     *  \retval     false   有効化されていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsEnabledAsynchronous() const noexcept
    {
        return _isAsynchronous;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      非同期実行時のバリアIDを取得
     */
    /* ------------------------------------------------------------------------- */
    template <typename TaskIDType = Identifier>
    [[nodiscard]] constexpr TaskIDType GetBarrierIdentifier() const noexcept
    {
        return TaskIDType(_barrierIdentifier);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      有効なバリアIDが設定されているかを取得
     *  \retval     true    バリアIDが有効
     *  \retval     false   バリアIDが無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsEnabledBarrier() const noexcept
    {
        return _barrierIdentifier > _identifier;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タスク生成時に呼び出される関数
     */
    /* ------------------------------------------------------------------------- */
    virtual void OnInitialize() noexcept
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タスクへのイベント通知
     *  \tparam     EventIDType イベントの種類を表す型
     *  \param[in]  event       通知するイベントの種類
     *  \param[in]  argument    通知先のタスクに渡す引数
     */
    /* ------------------------------------------------------------------------- */
    template <typename EventIDType>
    void NotifyEvent(EventIDType event, void *argument) noexcept
    {
        OnReceiveTaskEvent(EventIdentifier(event), argument);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タスク実行時に呼び出される関数
     *  \param[in]  stage   実行ステージ
     */
    /* ------------------------------------------------------------------------- */
    virtual void OnExecute(ExecuteStage stage) noexcept = 0;


protected:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タスクのイベント受信処理
     *  \param[in]  eventID     イベントID
     *  \param[in]  argument    引数
     */
    /* ------------------------------------------------------------------------- */
    virtual void OnReceiveTaskEvent(EventIdentifier eventID, void *argument) noexcept
    {
        (void)eventID;
        (void)argument;
    }

private:
    Identifier _identifier{0};
    ResideLevel _resideLevel{ResideLevel::NoResident};
    bool _isRequestedKill{false};
    bool _isAsynchronous{false};
    Identifier _barrierIdentifier{0};
};
}    // namespace MGL::Task

#endif    // INCGUARD_MGL_TASK_NODE_H_1622533593

// vim: et ts=4 sw=4 sts=4
