// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_task_defs.h
 *  \brief      MGL タスク関連定義
 *  \date       Since: June 3, 2021. 10:52:44 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TASK_DEFS_H_1622685164
#define INCGUARD_MGL_TASK_DEFS_H_1622685164

#include <cstddef>
#include <cstdint>
#include <type_traits>

#include <mgl/stl/mgl_stl_containers.h>
#include <mgl/system/mgl_system_debug_macro.h>

namespace MGL::Task
{
//! タスクの種類を表す型
using Identifier = uint32_t;

//! タスクのユニークIDを表す型
enum class UniqueIdentifier : uint64_t
{
    Invalid = 0,                                                        //!< 無効値
    Start = 1,                                                          //!< 開始
    End = ~static_cast<std::underlying_type_t<UniqueIdentifier>>(0),    //!< 終了
};

//! ユニークIDのプリインクリメントのオーバーロード
constexpr UniqueIdentifier operator++(UniqueIdentifier &rhs) noexcept
{
    rhs = UniqueIdentifier{static_cast<std::underlying_type_t<UniqueIdentifier>>(rhs) + 1};
    if (rhs >= UniqueIdentifier::End)
    {
        rhs = UniqueIdentifier::Start;
    }

    return rhs;
}

//! ユニークIDのポストインクリメントのオーバーロード
constexpr UniqueIdentifier operator++(UniqueIdentifier &rhs, int) noexcept
{
    auto ret = rhs;
    rhs = UniqueIdentifier{static_cast<std::underlying_type_t<UniqueIdentifier>>(rhs) + 1};
    if (rhs >= UniqueIdentifier::End)
    {
        rhs = UniqueIdentifier::Start;
    }

    return ret;
}

//! 実行モード
enum class ExecuteMode : uint8_t
{
    NormalUpdate,            //!< 通常更新
    ParallelizableUpdate,    //!< 並列化可能な更新
    RenderUpdate,            //!< 描画用の更新
};

//! 実行ステージ
using ExecuteStage = uint8_t;

//! 実行ステージの最大数
constexpr ExecuteStage kExecuteStageMax = 32;

//! 実行ステージ設定
struct StageSettings
{
    ExecuteStage stage;    //!< 実行ステージ
    ExecuteMode mode;      //!< 実行モード

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  inStage     実行ステージ
     *  \param[in]  inMode      実行モード
     */
    /* ------------------------------------------------------------------------- */
    constexpr StageSettings(ExecuteStage inStage, ExecuteMode inMode) noexcept
        : stage(inStage)
        , mode(inMode)
    {
        MGL_ASSERT(stage < kExecuteStageMax, "[MGL Task] Execute stage must be under %d.", kExecuteStageMax);
    }
};

//! 実行ステージ設定の配列
using StageSettingsArray = STL::vector<StageSettings>;

//! 初期化記述子
struct InitializeDescriptor
{
    StageSettingsArray stageSettings;
    int32_t parallelExecuteCount{0};
};

//! イベントIDを表す型
using EventIdentifier = uint32_t;

//! 常駐レベル
enum class ResideLevel : uint8_t
{
    NoResident = 0,    //!< 常駐しない
    Low = 0x40,        //!< 低
    Middle = 0x70,     //!< 中
    High = 0xB0,       //!< 高
    Max = 0xFF,        //!< 最高
};

}    // namespace MGL::Task

#endif    // INCGUARD_MGL_TASK_DEFS_H_1622685164

// vim: et ts=4 sw=4 sts=4
