// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_task_default_task_node.h
 *  \brief      MGL デフォルトタスク
 *  \date       Since: June 3, 2021. 17:03:51 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TASK_DEFAULT_TASK_NODE_H_1622707431
#define INCGUARD_MGL_TASK_DEFAULT_TASK_NODE_H_1622707431

#include <type_traits>

#include <mgl/task/mgl_task_node.h>

namespace MGL::Task
{
//! デフォルトの実行ステージ設定に対応したタスク
template <typename TaskIDType = Identifier, typename EventIDType = EventIdentifier>
class DefaultTaskNode : public Node
{
private:
    //! デフォルトの実行ステージ
    enum class Stage : uint8_t
    {
        Update,    //!< 更新処理
        Render,    //!< 描画処理
    };

public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このタスク用の初期化記述子を取得
     *  \param[in]  parallelExecuteCount    並列実行の最大数
     *  \return     初期化記述子
     */
    /* ------------------------------------------------------------------------- */
    static InitializeDescriptor GetInitializeDescriptor(int32_t parallelExecuteCount = -1) noexcept
    {
        InitializeDescriptor descriptor;
        descriptor.stageSettings =
            {
                {static_cast<ExecuteStage>(Stage::Update), ExecuteMode::ParallelizableUpdate},
                {static_cast<ExecuteStage>(Stage::Render), ExecuteMode::RenderUpdate},
            };
        descriptor.parallelExecuteCount = parallelExecuteCount;

        return descriptor;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  identifier    タスクのID
     */
    /* ------------------------------------------------------------------------- */
    explicit constexpr DefaultTaskNode(TaskIDType identifier) noexcept
        : Node(static_cast<Identifier>(identifier))
    {}

    ~DefaultTaskNode() noexcept override = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      実行時の処理
     *  \param[in]  stage   実行ステージ
     */
    /* ------------------------------------------------------------------------- */
    void OnExecute(ExecuteStage stage) noexcept final
    {
        switch (Stage{stage})
        {
            // 更新処理
            case Stage::Update:
                OnUpdate();
                break;

                // 描画処理
            case Stage::Render:
                OnRender();
                break;
        }
    }

protected:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      更新処理
     */
    /* ------------------------------------------------------------------------- */
    virtual void OnUpdate() noexcept {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      描画処理
     */
    /* ------------------------------------------------------------------------- */
    virtual void OnRender() noexcept {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      イベント受信処理
     *  \param[in]  event       発生したイベントの種類
     *  \param[in]  argument    イベントの引数
     */
    /* ------------------------------------------------------------------------- */
    virtual void OnEvent(EventIDType event, void *argument) noexcept
    {
        (void)event;
        (void)argument;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タスクのイベント受信処理
     *  \param[in]  eventID     イベントID
     *  \param[in]  argument    引数
     */
    /* ------------------------------------------------------------------------- */
    void OnReceiveTaskEvent(EventIdentifier eventID, void *argument) noexcept final
    {
        OnEvent(EventIDType(eventID), argument);
    }
};
}    // namespace MGL::Task
#endif    // INCGUARD_MGL_TASK_DEFAULT_TASK_NODE_H_1622707431

// vim: et ts=4 sw=4 sts=4
