// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_task.h
 *  \brief      MGL タスクシステム
 *  \date       Since: June 3, 2021. 11:46:50 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TASK_H_1622688410
#define INCGUARD_MGL_TASK_H_1622688410

#include <mgl/mgl_environment.h>
#include <mgl/task/mgl_task_default_task_node.h>
#include <mgl/task/mgl_task_node_list.h>
#include <mgl/task/mgl_task_weak_node.h>

namespace MGL::Task
{
bool Initialize(size_t capacity, const InitializeDescriptor &descriptor) noexcept;

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タスクの生成
 *  \tparam     TaskClass   生成するタスクのクラス
 *  \param[in]  args        コンストラクタに与える引数
 *  \return     生成されたタスクの弱参照ノード．失敗している場合は無効な弱参照ノード
 */
/* ------------------------------------------------------------------------- */
template <class TaskClass, class... Args>
constexpr WeakNode Create(Args... args) noexcept
{
    auto &nodeList = NodeList::GetInstance();
    if (!nodeList.HasFreeNode())
    {
        return {};
    }

    auto index = nodeList.Add(STL::make_unique<TaskClass>(args...));

    return WeakNode(index);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タスクの実行
 */
/* ------------------------------------------------------------------------- */
inline void Execute() noexcept
{
    NodeList::GetInstance().Execute();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      全てのタスクに削除要求を発行
 *  \param[in]  resideLevel    常駐レベルがこの値以下のタスクを削除する
 *  \return     削除要求を発行した数
 */
/* ------------------------------------------------------------------------- */
inline size_t Kill(ResideLevel resideLevel = ResideLevel::NoResident) noexcept
{
    return NodeList::GetInstance().Kill(resideLevel);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      指定したIDに対して削除要求を発行
 *  \tparam     TaskIDType      タスクの種類を表す型
 *  \param[in]  identifier      削除するID
 *  \param[in]  resideLevel     常駐レベルがこの値以下のタスクを削除する
 *  \return     削除要求を発行した数
 */
/* ------------------------------------------------------------------------- */
template <typename TaskIDType>
inline size_t Kill(TaskIDType identifier, ResideLevel resideLevel = ResideLevel::NoResident) noexcept
{
    return NodeList::GetInstance().Kill(Identifier(identifier), resideLevel);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      常駐プライオリティを無視して全てのタスクを削除
 *  \return     削除要求を発行した数
 */
/* ------------------------------------------------------------------------- */
inline size_t DestroyThemAll() noexcept
{
    return Kill(ResideLevel::Max);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      現在のタスクの総数を取得
 *  \return     現在のタスクの総数
 */
/* ------------------------------------------------------------------------- */
inline size_t GetCount() noexcept
{
    return NodeList::GetInstance().GetCount();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      指定したIDのタスクの数を取得
 *  \tparam     TaskIDType      タスクの種類を表す型
 *  \param[in]  identifier     数を取得するタスクの識別子
 *  \return     指定したIDのタスクの数
 */
/* ------------------------------------------------------------------------- */
template <typename TaskIDType>
inline size_t GetCount(TaskIDType identifier) noexcept
{
    return NodeList::GetInstance().GetCount(identifier);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タスクを検索
 *  \tparam     TaskIDType  タスクの種類を表す型
 *  \param[in]  identifier  検索するID
 *  \return     見つかったタスクの弱参照ノードの配列
 */
/* ------------------------------------------------------------------------- */
template <typename TaskIDType>
STL::vector<WeakNode> Find(TaskIDType identifier) noexcept
{
    STL::vector<WeakNode> result;

    NodeList::GetInstance().GetNodeArray(result, Identifier(identifier));

    return result;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タスクへのイベント通知
 *  \param[in]  eventID         イベントID
 *  \tparam     EventIDType     イベントの種類を表す型
 *  \param[in]  argument        引数
 */
/* ------------------------------------------------------------------------- */
template <typename EventIDType>
inline void NotifyEvent(EventIDType eventID, void *argument = nullptr) noexcept
{
    NodeList::GetInstance().NotifyEvent(EventIdentifier(eventID), argument);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タスクへのイベント通知
 *  \tparam     TaskIDType      タスクの種類を表す型
 *  \tparam     EventIDType     イベントの種類を表す型
 *  \param[in]  identifier      タスクID
 *  \param[in]  eventID         イベントID
 *  \param[in]  argument        引数
 */
/* ------------------------------------------------------------------------- */
template <typename TaskIDType, typename EventIDType>
inline void NotifyEvent(TaskIDType identifier, EventIDType eventID, void *argument = nullptr) noexcept
{
    NodeList::GetInstance().NotifyEvent(Identifier(identifier), EventIdentifier(eventID), argument);
}

}    // namespace MGL::Task

#endif    // INCGUARD_MGL_TASK_H_1622688410

// vim: et ts=4 sw=4 sts=4
