// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_window_module_interface.h
 *  \brief      MGL ウィンドウ機能のインターフェース
 *  \date       Since: February 19, 2021. 15:00:32 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_WINDOW_MODULE_INTERFACE_H_1613714432
#define INCGUARD_MGL_WINDOW_MODULE_INTERFACE_H_1613714432

#include <cstdint>

#include <mgl/system/mgl_system_defs.h>

namespace MGL::System
{
//! ウィンドウ機能のインターフェース
class WindowModuleInterface
{
public:
    virtual ~WindowModuleInterface() noexcept = default;


    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィンドウ機能が有効かを取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool IsAvailable() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      サイズの変更
     *  \param[in]  width   幅
     *  \param[in]  height  高さ
     */
    /* ------------------------------------------------------------------------- */
    virtual void Resize(uint32_t width, uint32_t height) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フルスクリーンの設定
     *  \param[in]  isEnabled   有効フラグ
     */
    /* ------------------------------------------------------------------------- */
    virtual void SetFullscreen(bool isEnabled) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フルスクリーンの有効・無効の切り替え
     */
    /* ------------------------------------------------------------------------- */
    virtual void ToggleFullscreen() noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フルスクリーン状態の取得
     *  \retval     true    フクスクリーン
     *  \retval     false   ウィンドウ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool IsFullscreen() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィンドウにフォーカスを設定
     *  \note       フォーカス = macOSでいうキーウィンドウ，Windowsでいうアクティブウィンドウ
     */
    /* ------------------------------------------------------------------------- */
    virtual void Focus() noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フォーカス状態の取得
     *  \retval     true    フォーカス
     *  \retval     false   フォーカスでない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool IsFocused() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アラートの表示
     *  \param[in]  message     表示メッセージ
     */
    /* ------------------------------------------------------------------------- */
    virtual void Alert(const char *message) const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィンドウタイトルの設定
     *  \param[in]  title   設定するウィンドウタイトル
     */
    /* ------------------------------------------------------------------------- */
    virtual void SetTitle(const char *title) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィンドウの識別子を取得
     *  \return     ウィンドウの識別子
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual int32_t GetIdentifier() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィンドウが存在している画面のサイズ情報を取得
     *  \return     画面サイズ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual ScreenSize GetScreenSize() const noexcept = 0;
};
}    // namespace MGL::System

#endif    // INCGUARD_MGL_WINDOW_MODULE_INTERFACE_H_1613714432

// vim: et ts=4 sw=4 sts=4
