// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_system_window.h
 *  \brief      MGL ウィンドウ制御
 *  \date       Since: February 20, 2021. 16:23:52 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_SYSTEM_WINDOW_H_1613805832
#define INCGUARD_MGL_SYSTEM_WINDOW_H_1613805832

#include <mgl/mgl_environment.h>
#include <mgl/system/mgl_system_module_set.h>

namespace MGL::System
{
//! MGL ウィンドウ制御クラス
class Window
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    Window() noexcept
        : _window(ModuleSet::GetInstance().GetWindowModule())
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      サイズの変更
     *  \param[in]  width   幅
     *  \param[in]  height  高さ
     */
    /* ------------------------------------------------------------------------- */
    void Resize(uint32_t width, uint32_t height) noexcept
    {
        _window.Resize(width, height);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フルスクリーンの設定
     *  \param[in]  isEnabled   有効フラグ
     */
    /* ------------------------------------------------------------------------- */
    void SetFullscreen(bool isEnabled) noexcept
    {
        _window.SetFullscreen(isEnabled);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フルスクリーンの有効・無効の切り替え
     */
    /* ------------------------------------------------------------------------- */
    void ToggleFullscreen() noexcept
    {
        _window.ToggleFullscreen();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フルスクリーン状態の取得
     *  \retval     true    フクスクリーン
     *  \retval     false   ウィンドウ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsFullscreen() const noexcept
    {
        return _window.IsFullscreen();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィンドウにフォーカスを設定
     *  \note       フォーカス = macOSでいうキーウィンドウ，Windowsでいうアクティブウィンドウ
     */
    /* ------------------------------------------------------------------------- */
    void Focus() noexcept
    {
        _window.Focus();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フォーカス状態の取得
     *  \retval     true    フォーカス
     *  \retval     false   フォーカスでない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsFocused() const noexcept
    {
        return _window.IsFocused();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アラートの表示
     *  \param[in]  message     表示メッセージ
     */
    /* ------------------------------------------------------------------------- */
    void Alert(const char *message) const noexcept
    {
        _window.Alert(message);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィンドウタイトルの設定
     *  \param[in]  title   設定するウィンドウタイトル
     */
    /* ------------------------------------------------------------------------- */
    void SetTitle(const char *title) noexcept
    {
        _window.SetTitle(title);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィンドウの識別子を取得
     *  \return     ウィンドウの識別子
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] int32_t GetIdentifier() const noexcept
    {
        return _window.GetIdentifier();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィンドウが存在しているディスプレイの情報を取得
     *  \return     ディスプレイ情報
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] ScreenSize GetScreenSize() const noexcept
    {
        return _window.GetScreenSize();
    }


private:
    WindowModuleInterface &_window;
};
}    // namespace MGL::System

#endif    // INCGUARD_MGL_SYSTEM_WINDOW_H_1613805832

// vim: et ts=4 sw=4 sts=4
