// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_system_module_set.h
 *  \brief      MGL システム関連モジュールセット
 *  \date       Since: February 19, 2021. 14:53:48 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_SYSTEM_MODULE_SET_H_1613714028
#define INCGUARD_MGL_SYSTEM_MODULE_SET_H_1613714028

#include <mgl/common/mgl_singleton.h>
#include <mgl/initialize/mgl_initializer_delegate.h>
#include <mgl/mgl_environment.h>
#include <mgl/system/application/mgl_application_module_interface.h>
#include <mgl/system/chrono/mgl_chrono_module_interface.h>
#include <mgl/system/debug/mgl_debug_module_interface.h>
#include <mgl/system/locale/mgl_locale_module_interface.h>
#include <mgl/system/window/mgl_window_module_interface.h>

namespace MGL::System
{
//! システム関連モジュールセットクラス
class ModuleSet final : public MGL::SharedSingleton<ModuleSet>
{
public:
    static STL::unique_ptr<ModuleSet> &GetInstanceRef() noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      初期化処理
     *  \param[in]  initializer     イニシャライザ
     */
    /* ------------------------------------------------------------------------- */
    bool Initialize(const InitializerDelegate &initializer) noexcept
    {
        _debug = initializer.MakeDebugModule();
        _locale = initializer.MakeLocaleModule();
        _chrono = initializer.MakeChronoModule();
        _window = initializer.MakeWindowModule();
        _application = initializer.MakeApplicationModule();

        if (_locale != nullptr)
        {
            _locale->Initialize();
        }

        return true;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィンドウ機能のモジュールを取得
     *  \return     ウィンドウ機能のモジュール
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] WindowModuleInterface &GetWindowModule() const noexcept
    {
        return *_window;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アプリケーション機能のモジュールを取得
     *  \return     アプリケーション機能のモジュール
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] ApplicationModuleInterface &GetApplicationModule() const noexcept
    {
        return *_application;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ロケール情報のモジュールを取得
     *  \return     ロケール情報のモジュール
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] LocaleModuleInterface &GetLocaleModule() const noexcept
    {
        return *_locale;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      日付と時間のモジュールを取得
     *  \return     日付と時間のモジュール
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] ChronoModuleInterface &GetChronoModule() const noexcept
    {
        return *_chrono;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デバッグモジュールを取得
     *  \return     デバッグモジュール
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] DebugModuleInterface &GetDebugModule() const noexcept
    {
        return *_debug;
    }

private:
    STL::unique_ptr<DebugModuleInterface> _debug;
    STL::unique_ptr<LocaleModuleInterface> _locale;
    STL::unique_ptr<ChronoModuleInterface> _chrono;
    STL::unique_ptr<WindowModuleInterface> _window;
    STL::unique_ptr<ApplicationModuleInterface> _application;
};
}    // namespace MGL::System

#endif    // INCGUARD_MGL_SYSTEM_MODULE_SET_H_1613714028

// vim: et ts=4 sw=4 sts=4
