// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_system_locale.h
 *  \brief      MGL ロケール情報
 *  \date       Since: February 24, 2021. 2:07:07 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_SYSTEM_LOCALE_H_1614100027
#define INCGUARD_MGL_SYSTEM_LOCALE_H_1614100027

#include <mgl/event/mgl_event.h>
#include <mgl/mgl_environment.h>
#include <mgl/system/mgl_system_chrono.h>
#include <mgl/system/mgl_system_module_set.h>

namespace MGL::System
{
//! ロケール情報クラス
class Locale
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    Locale() noexcept
        : _locale(ModuleSet::GetInstance().GetLocaleModule())
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      言語を取得
     *  \return     現在設定されている言語
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] Language GetLanguage() const noexcept
    {
        return _locale.GetLocaleInfo().language;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デフォルトの言語を取得
     *  \return     デフォルトの言語
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] Language GetDefaultLanguage() const noexcept
    {
        return _locale.GetDefaultLocaleInfo().language;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      言語を設定
     *  \param[in]  language    設定する言語
     */
    /* ------------------------------------------------------------------------- */
    void SetLanguage(Language language) noexcept
    {
        auto &currentLocaleInfo = _locale.GetLocaleInfo();
        if (currentLocaleInfo.language != language)
        {
            currentLocaleInfo.language = language;
            Event::Notify(Event::NotifyType::ChangeLanguage, &currentLocaleInfo.language);
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      時差を取得
     *  \return     時差（秒）
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] int32_t GetTimeDifference() const noexcept
    {
        return _locale.GetLocaleInfo().timeDifference;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デフォルトの時差を取得
     *  \return     デフォルトの時差（秒）
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] int32_t GetDefaultTimeDifference() const noexcept
    {
        return _locale.GetDefaultLocaleInfo().timeDifference;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      時差を設定
     *  \param[in]  timeDifference   設定する時差
     */
    /* ------------------------------------------------------------------------- */
    void SetTimeDifference(int32_t timeDifference) noexcept
    {
        _locale.GetLocaleInfo().timeDifference = timeDifference;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      現在のローカル時刻を取得
     *  \return     現在のローカル時刻
     */
    /* ------------------------------------------------------------------------- */
    DateTime GetLocalDateTime() noexcept
    {
        return {Chrono().GetCurrentEpochTime(), GetTimeDifference()};
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      桁区切りの間隔を取得
     *  \return     現在設定されている桁区切りの間隔
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] uint8_t GetGroupingSize() const noexcept
    {
        return _locale.GetLocaleInfo().groupingSize;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デフォルトの桁区切りの間隔を取得
     *  \return     デフォルトの桁区切りの間隔
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] uint8_t GetDefaultGroupingSize() const noexcept
    {
        return _locale.GetDefaultLocaleInfo().groupingSize;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      桁区切りの間隔を設定
     *  \param[in]  groupingSize    桁区切りの間隔
     */
    /* ------------------------------------------------------------------------- */
    void SetGroupingSize(uint8_t groupingSize) noexcept
    {
        _locale.GetLocaleInfo().groupingSize = groupingSize;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      数値の桁区切り記号を取得
     *  \return     現在設定されている桁区切り記号
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const char *GetGroupingSeparator() const noexcept
    {
        return _locale.GetLocaleInfo().groupingSeparator.c_str();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デフォルトの数値の桁区切り記号を取得
     *  \return     デフォルトの桁区切り記号
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const char *GetDefaultGroupingSeparator() const noexcept
    {
        return _locale.GetDefaultLocaleInfo().groupingSeparator.c_str();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      数値の桁区切り記号を設定
     *  \param[in]  separator   区切り記号
     */
    /* ------------------------------------------------------------------------- */
    void SetGroupingSeparator(const char *separator) noexcept
    {
        _locale.GetLocaleInfo().groupingSeparator = separator;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      小数点の区切り記号を取得
     *  \return     現在設定されている区切り記号
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const char *GetDecimalSeparator() const noexcept
    {
        return _locale.GetLocaleInfo().decimalSeparator.c_str();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デフォルトの小数点の区切り記号を取得
     *  \return     デフォルトの小数点の区切り記号
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const char *GetDefaultDecimalSeparator() const noexcept
    {
        return _locale.GetDefaultLocaleInfo().decimalSeparator.c_str();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      小数点の区切り記号を設定
     *  \param[in]  separator   区切り記号
     */
    /* ------------------------------------------------------------------------- */
    void SetDecimalSeparator(const char *separator) noexcept
    {
        _locale.GetLocaleInfo().decimalSeparator = separator;
    }

private:
    LocaleModuleInterface &_locale;
};
}    // namespace MGL::System

#endif    // INCGUARD_MGL_SYSTEM_LOCALE_H_1614100027

// vim: et ts=4 sw=4 sts=4
