// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_system_chrono.h
 *  \brief      MGL 日付と時間
 *  \date       Since: February 24, 2021. 5:14:19 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_SYSTEM_CHRONO_H_1614111259
#define INCGUARD_MGL_SYSTEM_CHRONO_H_1614111259

#include <mgl/mgl_environment.h>
#include <mgl/system/mgl_system_module_set.h>

namespace MGL::System
{
//! 日付と時間のクラス
class Chrono
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    Chrono() noexcept
        : _chrono(ModuleSet::GetInstance().GetChronoModule())
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      現在時刻をエポック時間で取得
     *  \return     現在時刻
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] EpochTime GetCurrentEpochTime() const noexcept
    {
        return _chrono.GetCurrentEpochTime();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      分を秒に変換
     *  \param[in]  minutes 変換する分
     *  \return     変換後の秒数
     */
    /* ------------------------------------------------------------------------- */
    static constexpr int32_t MinutesToSeconds(int32_t minutes) noexcept
    {
        return minutes * 60;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      時間を秒に変換
     *  \param[in]  hours 変換する時間
     *  \return     変換後の秒数
     */
    /* ------------------------------------------------------------------------- */
    static constexpr int32_t HoursToSeconds(int32_t hours) noexcept
    {
        return MinutesToSeconds(hours * 60);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      日数を秒に変換
     *  \param[in]  days 変換する日数
     *  \return     変換後の秒数
     */
    /* ------------------------------------------------------------------------- */
    static constexpr int32_t DaysToSeconds(int32_t days) noexcept
    {
        return HoursToSeconds(days * 24);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      Tickタイマーに対応しているかを取得
     *  \retval     true    対応している
     *  \retval     false   対応していない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsAvailableTickTimer() const noexcept
    {
        return _chrono.IsAvailableTickTimer();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      現在のTickTimeを取得
     *  \return     現在のTickTime
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] TickTime GetTickTime() const noexcept
    {
        return _chrono.GetTickTime();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      2つのTickTimeから経過時間を取得
     *  \param[in]  start    開始時間
     *  \param[in]  end      終了時間
     *  \return     開始時間から終了時間までの経過時間
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] static constexpr TickTime GetTickPassTime(TickTime start, TickTime end)
    {
        if (start < end)
        {
            return end - start;
        }
        else if (start > end)
        {
            return static_cast<TickTime>(~0u) - start + end;
        }
        else
        {
            return 0;
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      TickTimeをミリ秒に変換
     *  \return     変換後の時間
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] float TicksToMilliseconds(TickTime tickTime) const noexcept
    {
        return _chrono.TicksToNanoseconds(tickTime) / 1000.0f / 1000.0f;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      TickTimeをマイクロ秒に変換
     *  \return     変換後の時間
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] float TicksToMicroseconds(TickTime tickTime) const noexcept
    {
        return _chrono.TicksToNanoseconds(tickTime) / 1000.0f;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      TickTimeをナノ秒に変換
     *  \return     変換後の時間
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] float TicksToNanoseconds(TickTime tickTime) const noexcept
    {
        return _chrono.TicksToNanoseconds(tickTime);
    }

private:
    ChronoModuleInterface &_chrono;
};
}    // namespace MGL::System

#endif    // INCGUARD_MGL_SYSTEM_CHRONO_H_1614111259

// vim: et ts=4 sw=4 sts=4
