// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_system_application.h
 *  \brief      MGL アプリケーション制御
 *  \date       Since: February 22, 2021. 12:12:27 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_SYSTEM_APPLICATION_H_1613963547
#define INCGUARD_MGL_SYSTEM_APPLICATION_H_1613963547

#include <mgl/mgl_environment.h>
#include <mgl/system/mgl_system_module_set.h>

namespace MGL::System
{
//! アプリケーション制御クラス
class Application
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    Application() noexcept
        : _application(ModuleSet::GetInstance().GetApplicationModule())
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スリープの抑制
     *  \param[in]  isEnabled   有効フラグ
     *  \retval     true        成功
     *  \retval     false       失敗
     */
    /* ------------------------------------------------------------------------- */
    bool Caffeinate(bool isEnabled) noexcept
    {
        return _application.Caffeinate(isEnabled);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スリープの抑制状態を取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsCaffeinated() const noexcept
    {
        return _application.IsCaffeinated();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      実行引数を取得
     *  \return     実行引数の配列
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const ArgumentArray &GetArguments() const noexcept
    {
        return _application.GetArguments();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アプリケーションの終了
     */
    /* ------------------------------------------------------------------------- */
    void Quit() noexcept
    {
        _application.Quit();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アプリケーションの終了をサポートしているかを取得
     *  \retval     true    サポートしている
     *  \retval     false   サポートしていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsAvailableQuit() const noexcept
    {
        return _application.IsAvailableQuit();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      前回のフレームからの経過時間を取得
     *  \return     経過時間（ミリ秒）
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] float GetFrameElapsedTime() const noexcept
    {
        return _application.GetFrameElapsedTime();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      FPSを取得
     *  \return     FPS（過去1秒間に実際に更新された回数）
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] uint32_t GetFramesPerSecond() const noexcept
    {
        return _application.GetFramesPerSecond();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      クリップボードにテキストをコピー
     *  \param[in]  text    コピーするテキスト
     *  \retval     true    成功
     *  \retval     false   失敗
     */
    /* ------------------------------------------------------------------------- */
    bool SetClipboardText(const char *text) noexcept
    {
        return _application.SetClipboardText(text);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      クリップボードからテキストを取得
     *  \param[out] text    テキストのコピー先
     *  \retval     true    成功
     *  \retval     false   失敗
     */
    /* ------------------------------------------------------------------------- */
    bool GetClipboardText(STL::string &text) noexcept
    {
        return _application.GetClipboardText(text);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      システムがクリップボードをサポートしているかを取得
     *  \retval     true    サポートしている
     *  \retval     false   サポートしていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsAvailableClipboard() const noexcept
    {
        return _application.IsAvailableClipboard();
    }

private:
    ApplicationModuleInterface &_application;
};

}    // namespace MGL::System

#endif    // INCGUARD_MGL_SYSTEM_APPLICATION_H_1613963547

// vim: et ts=4 sw=4 sts=4
