// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_chrono_defs.h
 *  \brief      MGL 日付と時間の定義
 *  \date       Since: February 24, 2021. 4:19:19 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_CHRONO_DEFS_H_1614107959
#define INCGUARD_MGL_CHRONO_DEFS_H_1614107959

#include <cstdint>
#include <ctime>

#include <mgl/mgl_environment.h>

namespace MGL::System
{
//! UNIXエポック時間を表す型
using EpochTime = time_t;
static_assert(sizeof(EpochTime) >= 8);    // 64ビット以上を必須にする

//! 日付と時刻
struct DateTime
{
    uint16_t year;             //!< 年
    uint8_t month;             //!< 月
    uint8_t day;               //!< 日
    uint8_t hour;              //!< 時間
    uint8_t minute;            //!< 分
    uint8_t second;            //!< 秒
    int32_t timeDifference;    //!< 時差（秒）

    //! コンストラクタ
    constexpr DateTime() noexcept
        : year(1970)
        , month(1)
        , day(1)
        , hour(0)
        , minute(0)
        , second(0)
        , timeDifference(0)
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エポック時間から初期化
     *  \param[in]  epochTime           エポック時間
     *  \param[in]  inTimeDifference    時差（秒）
     */
    /* ------------------------------------------------------------------------- */
    DateTime(EpochTime epochTime, int32_t inTimeDifference = 0) noexcept
        : timeDifference(inTimeDifference)
    {
        epochTime += inTimeDifference;

        // TODO:
#if defined(MGL_TARGET_WIN32)
        struct tm temp;
        gmtime_s(&temp, &epochTime);
        struct tm *time = &temp;
#else
        struct tm *time = gmtime(&epochTime);
#endif

        year = static_cast<uint16_t>(time->tm_year + 1900);
        month = static_cast<uint8_t>(time->tm_mon + 1);
        day = static_cast<uint8_t>(time->tm_mday);
        hour = static_cast<uint8_t>(time->tm_hour);
        minute = static_cast<uint8_t>(time->tm_min);
        second = static_cast<uint8_t>(time->tm_sec);
    }
};

//! Tickタイムを表す型
using TickTime = uint64_t;

}    // namespace MGL::System

#endif    // INCGUARD_MGL_CHRONO_DEFS_H_1614107959

// vim: et ts=4 sw=4 sts=4
