// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_application_module_interface.h
 *  \brief      MGL アプリケーション機能のインターフェース
 *  \date       Since: February 22, 2021. 9:40:29 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_APPLICATION_MODULE_INTERFACE_H_1613954429
#define INCGUARD_MGL_APPLICATION_MODULE_INTERFACE_H_1613954429

#include <mgl/mgl_application_delegate.h>
#include <mgl/stl/mgl_stl_containers.h>
#include <mgl/stl/mgl_stl_memory.h>
#include <mgl/stl/mgl_stl_string.h>

namespace MGL::System
{
//! 引数配列の型
using ArgumentArray = STL::vector<STL::string>;

//! アプリケーションモジュールのインターフェース
class ApplicationModuleInterface
{
public:
    virtual ~ApplicationModuleInterface() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スリープの抑制
     *  \param[in]  isEnabled   有効フラグ
     *  \retval     true        成功
     *  \retval     false       失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool Caffeinate(bool isEnabled) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スリープの抑制状態を取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool IsCaffeinated() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      実行引数を取得
     *  \return     実行引数の配列
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual const ArgumentArray &GetArguments() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アプリケーションの終了
     */
    /* ------------------------------------------------------------------------- */
    virtual void Quit() noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アプリケーションの終了をサポートしているかを取得
     *  \retval     true    サポートしている
     *  \retval     false   サポートしていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool IsAvailableQuit() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      前回のフレームからの経過時間を取得
     *  \return     経過時間（ミリ秒）
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual float GetFrameElapsedTime() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      FPSを取得
     *  \return     FPS（過去1秒間に実際に更新された回数）
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual uint32_t GetFramesPerSecond() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アプリケーションデリゲートの登録
     *  \param[in]  delegate    登録するアプリケーションデリゲート
     */
    /* ------------------------------------------------------------------------- */
    virtual void SetApplicationDelegate(STL::unique_ptr<ApplicationDelegate> delegate) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      クリップボードにテキストをコピー
     *  \param[in]  text    コピーするテキスト
     *  \retval     true    成功
     *  \retval     false   失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool SetClipboardText(const char *text) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      クリップボードからテキストを取得
     *  \param[out] text    テキストのコピー先
     *  \retval     true    成功
     *  \retval     false   失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool GetClipboardText(STL::string &text) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      システムがクリップボードをサポートしているかを取得
     *  \retval     true    サポートしている
     *  \retval     false   サポートしていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool IsAvailableClipboard() const noexcept = 0;
};
}    // namespace MGL::System

#endif    // INCGUARD_MGL_APPLICATION_MODULE_INTERFACE_H_1613954429

// vim: et ts=4 sw=4 sts=4
