// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_savedata_server_delegate.h
 *  \brief      MGL セーブデータサーバのデリゲート
 *  \date       Since: August 5, 2021. 8:43:16 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_SAVEDATA_SERVER_DELEGATE_H_1628120596
#define INCGUARD_MGL_SAVEDATA_SERVER_DELEGATE_H_1628120596

#include <mgl/savedata/mgl_savedata_defs.h>

namespace MGL::Savedata
{
//! セーブデータサーバのデリゲート
class ServerDelegate
{
public:
    virtual ~ServerDelegate() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      初期化処理
     *  \return     0で成功．0以外の正の数でデリゲート側定義のエラーコード（負の数はサーバ側の予約）
     */
    /* ------------------------------------------------------------------------- */
    virtual int32_t OnInitialize() noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ファイルアクセス前に実行される処理
     *  \param[in]  identifier  セーブデータの識別子
     *  \param[in]  index       インデックス（用途はデリゲート側で定義）
     *  \param[in]  accessType  アクセスタイプ
     *  \retval     true        成功
     *  \retval     false       失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool OnPreAccess(DataIdentifier identifier, uint32_t index, AccessType accessType) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ファイルアクセス後に実行される処理
     *  \param[in]  identifier  セーブデータの識別子
     *  \param[in]  index       インデックス（用途はデリゲート側で定義）
     *  \param[in]  accessType  アクセスタイプ
     *  \retval     true        成功
     *  \retval     false       失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool OnPostAccess(DataIdentifier identifier, uint32_t index, AccessType accessType) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ファイルの情報を取得
     *  \param[in]  identifier  セーブデータの識別子
     *  \param[in]  index       インデックス（用途はデリゲート側で定義）
     *  \return     引数に対応したファイルの情報
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual const FileInfo *GetFileInfo(DataIdentifier identifier, uint32_t index) const noexcept = 0;
};
}    // namespace MGL::Savedata

#endif    // INCGUARD_MGL_SAVEDATA_SERVER_DELEGATE_H_1628120596

// vim: et ts=4 sw=4 sts=4
