// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_savedata_server.h
 *  \brief      MGL セーブデータサーバ
 *  \date       Since: August 5, 2021. 9:18:18 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_SAVEDATA_SERVER_H_1628122698
#define INCGUARD_MGL_SAVEDATA_SERVER_H_1628122698

#include <future>

#include <mgl/common/mgl_singleton.h>
#include <mgl/savedata/mgl_savedata_chunk.h>
#include <mgl/savedata/mgl_savedata_server_delegate.h>
#include <mgl/stl/mgl_stl_containers.h>
#include <mgl/stl/mgl_stl_memory.h>

namespace MGL::Savedata
{
//! セーブデータサーバ
class Server final : public MGL::SharedSingleton<Server>
{
public:
    static STL::unique_ptr<Server> &GetInstanceRef() noexcept;

    Server() noexcept;

    int32_t Initialize(STL::unique_ptr<ServerDelegate> &delegate) noexcept;

    bool AddChunk(DataIdentifier identifier, Chunk *chunk) noexcept;
    bool RemoveChunk(DataIdentifier identifier, Chunk *chunk) noexcept;

    bool SaveRequest(const STL::vector<RequestInfo> &requests, bool async, bool haltOnError) noexcept;
    bool LoadRequest(const STL::vector<RequestInfo> &requests, bool async, bool haltOnError, bool errorOnFileNotExist) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ロード中かを取得
     *  \retval     true    ロード中
     *  \retval     false   ロード中でない
     */
    /* ------------------------------------------------------------------------- */
    bool IsLoading() const noexcept
    {
        return _state == State::Loading;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      セーブ中かを取得
     *  \retval     true    セーブ中
     *  \retval     false   セーブ中でない
     */
    /* ------------------------------------------------------------------------- */
    bool IsSaving() const noexcept
    {
        return _state == State::Saving;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      セーブ・ロードが可能な状態かを取得
     *  \retval     true    可能
     *  \retval     false   不可能（処理中 or 未初期化）
     */
    /* ------------------------------------------------------------------------- */
    bool IsReady() const noexcept
    {
        return _state == State::Ready;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      処理中かを取得
     *  \retval     true    処理中
     *  \retval     false   処理中でない
     */
    /* ------------------------------------------------------------------------- */
    bool IsProcessing() const noexcept
    {
        return IsLoading() || IsSaving();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      前回の処理に成功したかを取得
     *  \retval     true    成功
     *  \retval     false   失敗 or 処理中
     */
    /* ------------------------------------------------------------------------- */
    bool IsSucceeded() const noexcept
    {
        return IsReady() && !_existError;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      最後に発生した処理結果を取得
     *  \return     処理結果
     */
    /* ------------------------------------------------------------------------- */
    constexpr const STL::vector<Result> &GetLastResults() const noexcept
    {
        return _lastResults;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      実行中の処理が完了するまで待機
     *  \retval     true    処理に成功
     *  \retval     false   処理に失敗
     */
    /* ------------------------------------------------------------------------- */
    bool Wait() noexcept
    {
        return _future.get();
    }

private:
    //! 状態
    enum class State : uint8_t
    {
        PreInitialize,    //!< 初期化前
        Ready,            //!< 準備完了
        Loading,          //!< 読み込み中
        Saving,           //!< 書き込み中
    };

    Result Save(DataIdentifier identifier, uint32_t index) noexcept;
    Result Load(DataIdentifier identifier, uint32_t index) noexcept;

    STL::unique_ptr<ServerDelegate> _delegate;

    using ChunkList = STL::list<Chunk *>;
    using ChunkListMap = STL::unordered_map<DataIdentifier, ChunkList>;

    ChunkListMap _chunkListMap;

    std::atomic<State> _state;
    std::future<bool> _future;

    STL::vector<RequestInfo> _requests;

    STL::vector<Result> _lastResults;
    std::atomic<bool> _existError;
};
}    // namespace MGL::Savedata

#endif    // INCGUARD_MGL_SAVEDATA_SERVER_H_1628122698

// vim: et ts=4 sw=4 sts=4
