// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_savedata_defs.h
 *  \brief      MGL セーブデータ関連定義
 *  \date       Since: August 5, 2021. 8:44:04 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_SAVEDATA_DEFS_H_1628120644
#define INCGUARD_MGL_SAVEDATA_DEFS_H_1628120644

#include <cstddef>
#include <cstdint>

#include <mgl/file/mgl_file.h>
#include <mgl/stl/mgl_stl_string.h>

namespace MGL::Savedata
{
//! アクセスタイプ
enum class AccessType : uint8_t
{
    Save,    //!< セーブ
    Load     //!< ロード
};

//! セーブデータ識別子
using DataIdentifier = uint32_t;

//! チャンク識別子
using ChunkIdentifier = uint32_t;

//! ファイル情報
struct FileInfo
{
    const File::PathView path;    //!< パス
    size_t size;                  //!< サイズ
};

//! エラーの種類
enum class Error : uint8_t
{
    None,                  //!< エラーなし
    NoOperation,           //!< 処理していない
    BadImplement,          //!< 実装に誤りがある
    DelegateNotReady,      //!< デリゲートが準備できていない
    DelegateHasError,      //!< デリゲートでエラーが発生
    IdentifierNotFound,    //!< 識別子が見つからない
    BufferNotEnough,       //!< バッファが足りない
    FailedToFileAccess,    //!< ファイルアクセスの際にエラーが発生
    MarkerNotMatching,     //!< マーカーの不一致
    FailedToChunkParse,    //!< チャンクのパースに失敗
    Unknown                //!< 不明
};

//! リクエスト情報
struct RequestInfo
{
    DataIdentifier identifier;    //!< データ識別子
    uint32_t index;               //!< インデックス

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  inIdentifier    データ識別子
     *  \param[in]  inIndex         インデックス
     */
    /* ------------------------------------------------------------------------- */
    constexpr RequestInfo(DataIdentifier inIdentifier, uint32_t inIndex = 0) noexcept
        : identifier(inIdentifier)
        , index(inIndex)
    {
    }
};

//! 処理結果
struct Result
{
    DataIdentifier identifier;      //!< セーブデータの識別子
    Error error;                    //!< エラーの種類
    File::Error fileAccessError;    //!< ファイルアクセスで発生したエラーの種類（typeがFailedToFileAccessの際に有効）

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  inIdentifier        セーブデータの識別子
     *  \param[in]  inError             エラーの種類
     *  \param[in]  inFileAccessError   ファイルアクセスで発生したエラー
     */
    /* ------------------------------------------------------------------------- */
    constexpr Result(DataIdentifier inIdentifier, Error inError, File::Error inFileAccessError = File::Error::None) noexcept
        : identifier(inIdentifier)
        , error(inError)
        , fileAccessError(inFileAccessError)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr Result() noexcept
        : Result(0, Error::NoOperation, File::Error::None)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      bool型のキャスト
     *  \retval     true    エラーなし
     *  \retval     false   エラーが発生している
     */
    /* ------------------------------------------------------------------------- */
    explicit constexpr operator bool() const noexcept
    {
        return error == Error::None;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      否定演算子
     *  \retval     true    エラーが発生している
     *  \retval     false   エラーなし
     */
    /* ------------------------------------------------------------------------- */
    constexpr bool operator!() const noexcept
    {
        return !static_cast<bool>(*this);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エラーが発生しているかを取得
     *  \param[in]  errorOnFileNotExist     trueを指定した場合，ファイルが存在しない状態をエラー扱いにする
     *  \retval     true                    エラーが発生している
     *  \retval     false                   エラーなし
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool HasError(bool errorOnFileNotExist = true) const noexcept
    {
        if (error == Error::None)
        {
            return false;
        }

        if (!errorOnFileNotExist)
        {
            if ((error == Error::FailedToFileAccess) && (fileAccessError == File::Error::FileNotExist))
            {
                return false;
            }
        }

        return true;
    }
};

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      日付からバージョンに変換
 *  \param[in]  year        年
 *  \param[in]  month       月
 *  \param[in]  day         日
 *  \param[in]  revision    リビジョン
 *  \return     引数から計算されたバージョン番号
 */
/* ------------------------------------------------------------------------- */
constexpr uint32_t MakeDateVersion(uint32_t year, uint32_t month, uint32_t day, uint32_t revision = 0)
{
    return (year * 1'000'000) + (month * 10'000) + (day * 100) + revision;
}
}    // namespace MGL::Savedata
#endif    // INCGUARD_MGL_SAVEDATA_DEFS_H_1628120644

// vim: et ts=4 sw=4 sts=4
