// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_savedata.h
 *  \brief      MGL セーブデータ
 *  \date       Since: August 11, 2021. 10:17:30 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_SAVEDATA_H_1628644650
#define INCGUARD_MGL_SAVEDATA_H_1628644650

#include <mgl/savedata/mgl_savedata_chunk_register.h>
#include <mgl/savedata/mgl_savedata_server.h>

namespace MGL::Savedata
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      セーブデータサーバの生成
 *  \retval     0           成功
 *  \retval     0以外の正数 デリゲートの初期化エラー（意味はデリゲート側で定義）
 *  \retval     負数        サーバ側の要因によるエラー（現状は-1のみ，他は予約）
 */
/* ------------------------------------------------------------------------- */
template <class DelegateClass, class... Args>
inline int32_t CreateServer(Args... args) noexcept
{
    if (Server::HasInstance())
    {
        return false;
    }

    auto &server = Server::CreateInstance();

    STL::unique_ptr<ServerDelegate> delegate = STL::make_unique<DelegateClass>(args...);

    return server.Initialize(delegate);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      セーブデータサーバの取得
 *  \return     セーブデータサーバ．存在しない場合はnullptr
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] inline Server *GetServer() noexcept
{
    if (Server::HasInstance())
    {
        return &Server::GetInstance();
    }

    return nullptr;
}

bool SaveRequests(const STL::vector<RequestInfo> &requests, bool haltOnError = true) noexcept;
bool SaveRequests(DataIdentifier identifier, uint32_t index = 0) noexcept;
bool LoadRequests(const STL::vector<RequestInfo> &requests, bool haltOnError = true, bool errorOnFileNotExist = true) noexcept;
bool LoadRequests(DataIdentifier identifier, uint32_t index = 0, bool errorOnFileNotExist = true) noexcept;

bool SaveSynchronously(const STL::vector<RequestInfo> &requests, bool haltOnError = true) noexcept;
bool SaveSynchronously(DataIdentifier identifier, uint32_t index = 0) noexcept;
bool LoadSynchronously(const STL::vector<RequestInfo> &requests, bool haltOnError = true, bool errorOnFileNotExist = true) noexcept;
bool LoadSynchronously(DataIdentifier identifier, uint32_t index = 0, bool errorOnFileNotExist = true) noexcept;

[[nodiscard]] bool IsLoading() noexcept;
[[nodiscard]] bool IsSaving() noexcept;
[[nodiscard]] bool IsReady() noexcept;
[[nodiscard]] bool IsProcessing() noexcept;
[[nodiscard]] bool IsSucceeded() noexcept;
[[nodiscard]] STL::vector<Result> GetLastResults() noexcept;
bool Wait() noexcept;
}    // namespace MGL::Savedata

#endif    // INCGUARD_MGL_SAVEDATA_H_1628644650

// vim: et ts=4 sw=4 sts=4
