// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_texture_storage.h
 *  \brief      MGL テクスチャストレージ
 *  \date       Since: December 10, 2020. 16:03:25 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TEXTURE_STORAGE_H_1607583805
#define INCGUARD_MGL_TEXTURE_STORAGE_H_1607583805

#include <mgl/common/mgl_singleton.h>
#include <mgl/hash/mgl_hash_fnv1a.h>
#include <mgl/render/mgl_texture_generator.h>
#include <mgl/render/mgl_texture_loader.h>
#include <mgl/render/mgl_texture_resource.h>
#include <mgl/stl/mgl_stl_containers.h>

#include <condition_variable>
#include <mutex>

namespace MGL::Render
{
//! テクスチャにアクセスするキーの型
enum class TextureKey : uint32_t {};

//! テクスチャキーを生成する際のハッシュのシード値
constexpr uint32_t kDefaultTextureKeySeed = MGL::Hash::kFNV1aDefaultValue32;    // 32bitの値なら何でもいい．衝突する場合は変更を

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャキーを生成
 *  \param[in]  key     キーの文字列
 *  \param[in]  seed    シード値（省略時はkDefaultTextureKeySeed）
 *  \return     キーから生成したハッシュ値
 */
/* ------------------------------------------------------------------------- */
constexpr TextureKey MakeTextureKey(const char *key, uint32_t seed = kDefaultTextureKeySeed) noexcept
{
    return TextureKey{Hash::FNV1a(key, seed)};
}


//! テクスチャストレージ
class TextureStorage
{
public:
    //! デフォルトのテクスチャローダーを表すキー
    static constexpr TextureLoaderKey kDefaultLoaderKey = MakeTextureLoaderKey("MGL-DefaultLoader");

    explicit TextureStorage(STL::unique_ptr<TextureGenerator> generator) noexcept;
    ~TextureStorage() noexcept;

    [[nodiscard]] SharedTextureResource Load(const void *imageData, size_t dataSize, TextureLoaderKey loaderKey = kDefaultLoaderKey) noexcept;
    [[nodiscard]] SharedTextureResource LoadAsync(const void *imageData, size_t dataSize, TextureLoaderKey loaderKey = kDefaultLoaderKey) noexcept;
    [[nodiscard]] SharedTextureResource Load(const File::PathView &imagePath, TextureLoaderKey loaderKey = kDefaultLoaderKey) noexcept;
    [[nodiscard]] SharedTextureResource LoadAsync(const File::PathView &imagePath, TextureLoaderKey loaderKey = kDefaultLoaderKey) noexcept;

    SharedTextureResource Load(TextureKey key, const void *imageData, size_t dataSize, TextureLoaderKey loaderKey = kDefaultLoaderKey) noexcept;
    SharedTextureResource LoadAsync(TextureKey key, const void *imageData, size_t dataSize, TextureLoaderKey loaderKey = kDefaultLoaderKey) noexcept;
    SharedTextureResource Load(TextureKey key, const File::PathView &imagePath, TextureLoaderKey loaderKey = kDefaultLoaderKey) noexcept;
    SharedTextureResource LoadAsync(TextureKey key, const File::PathView &imagePath, TextureLoaderKey loaderKey = kDefaultLoaderKey) noexcept;

    SharedTextureResource Create(TextureKey key, const void *pixelData, PixelFormat pixelFormat, uint32_t width, uint32_t height) noexcept;
    [[nodiscard]] SharedTextureResource Create(const void *pixelData, PixelFormat pixelFormat, uint32_t width, uint32_t height) noexcept;

    SharedTextureResource CreateRenderTarget(TextureKey key, uint32_t width, uint32_t height) noexcept;
    [[nodiscard]] SharedTextureResource CreateRenderTarget(uint32_t width, uint32_t height) noexcept;

    SharedTextureResource Get(TextureKey key) noexcept;

    bool Destroy(TextureKey key) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デフォルトで使用するローダーを設定
     *  \param[in]  loaderKey    設定するローダーのキー
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetDefaultLoader(TextureLoaderKey loaderKey) noexcept
    {
        _defaultLoaderKey = loaderKey;
    }

    bool RegisterLoader(TextureLoaderKey loaderKey, UniqueTextureLoader loader) noexcept;

private:
    TextureLoader *GetTextureLoader(TextureLoaderKey loaderKey) const noexcept;
    SharedTextureResource CreateDerivedResource() const noexcept;

    void SetLoading(TextureResource *resource, bool isLoading) noexcept;
    void WaitLoading() noexcept;

    STL::unique_ptr<TextureGenerator> _textureGenerator;
    STL::unordered_map<TextureKey, SharedTextureResource> _resources;
    STL::unordered_map<TextureLoaderKey, UniqueTextureLoader> _loaders;
    TextureLoaderKey _defaultLoaderKey{kDefaultLoaderKey};

    std::atomic<uint32_t> _loadingCount;
    std::condition_variable _loadingCondition;
    std::mutex _loadingMutex;

    std::mutex _tableMutex;
};
}    // namespace MGL::Render
#endif    // INCGUARD_MGL_TEXTURE_STORAGE_H_1607583805

// vim: et ts=4 sw=4 sts=4
