// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_texture_resource.h
 *  \brief      MGL テクスチャリソース
 *  \date       Since: December 8, 2020. 15:23:15 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TEXTURE_RESOURCE_H_1607408595
#define INCGUARD_MGL_TEXTURE_RESOURCE_H_1607408595

#include <atomic>
#include <cstdint>
#include <memory>

#include <mgl/math/mgl_vector2.h>
#include <mgl/render/mgl_render_types.h>

namespace MGL::Render
{
//! テクスチャリソース
class TextureResource
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr TextureResource() noexcept
        : _isLoading(false)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デストラクタ
     */
    /* ------------------------------------------------------------------------- */
    virtual ~TextureResource() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      リソースがどのレンダラ用のものかを返す
     *  \return     描画エンジンの種類
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual RendererType GetRendererType() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      有効状態の取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool IsValid() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      レンダーターゲットに設定可能かを取得
     *  \retval     true    設定可能
     *  \retval     false   不可能
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool IsRenderTarget() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      テクスチャの生成
     *  \param[in]  pixelData   ピクセルデータ
     *  \param[in]  pixelFormat ピクセルフォーマット
     *  \param[in]  width       幅
     *  \param[in]  height      高さ
     *  \retval     true        成功
     *  \retval     false       失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool Create(const void *pixelData, PixelFormat pixelFormat, uint32_t width, uint32_t height) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      レンダーターゲットの生成
     *  \param[in]  width       幅
     *  \param[in]  height      高さ
     *  \retval     true        成功
     *  \retval     false       失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool CreateRenderTarget(uint32_t width, uint32_t height) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      リソースの破棄
     */
    /* ------------------------------------------------------------------------- */
    virtual void Destroy() noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      テクスチャのサイズを取得
     *  \return     テクスチャのサイズ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual Vector2 GetSize() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      テクスチャのピクセルフォーマットを取得
     *  \return     テクスチャのピクセルフォーマット
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual PixelFormat GetPixelFormat() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      読み込み中かを取得
     *  \retval     true    読み込み中
     *  \retval     false   読み込み中でない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsLoading() const noexcept
    {
        return _isLoading;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      読み込み中かを設定
     *  \param[in]  isLoading   設定する読み込み中フラグ
     */
    /* ------------------------------------------------------------------------- */
    void SetLoading(bool isLoading) noexcept
    {
        _isLoading = isLoading;
    }

private:
    std::atomic<bool> _isLoading;
};

//! 共有するテクスチャリソースの型
using SharedTextureResource = std::shared_ptr<MGL::Render::TextureResource>;

}    // namespace MGL::Render
#endif    // INCGUARD_MGL_TEXTURE_RESOURCE_H_1607408595

// vim: et ts=4 sw=4 sts=4
