// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_texture_loader.h
 *  \brief      MGL テクスチャローダー
 *  \date       Since: December 12, 2020. 22:14:38 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TEXTURE_LOADER_H_1607778878
#define INCGUARD_MGL_TEXTURE_LOADER_H_1607778878

#include <mgl/file/mgl_file_path_view.h>
#include <mgl/hash/mgl_hash_fnv1a.h>
#include <mgl/mgl_environment.h>
#include <mgl/render/mgl_texture_resource.h>
#include <mgl/stl/mgl_stl_memory.h>

namespace MGL::Render
{
//! テクスチャローダーのキー
using TextureLoaderKey = uint32_t;

//! テクスチャローダーのキーを生成する際のシード値
constexpr uint32_t kDefaultTextureLoaderKeySeed = MGL::Hash::kFNV1aDefaultValue32;    // 32bitの値なら何でもいい．衝突する場合は変更を

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャローダーのキーを生成
 *  \param[in]  key     キーの文字列
 *  \param[in]  seed    シード値（省略時はkDefaultTextureLoaderKeySeed）
 *  \return     キーから生成したハッシュ値
 */
/* ------------------------------------------------------------------------- */
constexpr TextureLoaderKey MakeTextureLoaderKey(const char *key, uint32_t seed = kDefaultTextureLoaderKeySeed) noexcept
{
    return Hash::FNV1a(key, seed);
}

//! テクスチャローダー
class TextureLoader
{
public:
    virtual ~TextureLoader() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ファイルから読み込み
     *  \param[out] textureResource     読み込み先のテクスチャリソース
     *  \param[in]  path                ファイルのパス
     *  \retval     true                成功
     *  \retval     false               失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool Load(TextureResource &textureResource, const File::PathView &path) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      メモリ上のバッファから読み込み
     *  \param[out] textureResource     読み込み先のテクスチャリソース
     *  \param[in]  imageData           読み込むイメージデータ
     *  \param[in]  dataSize            読み込むイメージデータのサイズ
     *  \retval     true                成功
     *  \retval     false               失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool Load(TextureResource &textureResource, const void *imageData, size_t dataSize) noexcept = 0;
};

//! テクスチャローダーのユニークポインタ
using UniqueTextureLoader = STL::unique_ptr<TextureLoader>;
}    // namespace MGL::Render

#endif    // INCGUARD_MGL_TEXTURE_LOADER_H_1607778878

// vim: et ts=4 sw=4 sts=4
